<?php
/**
 * Template HTML - Interface Web
 * Arquivo: template.php
 */

require_once __DIR__ . '/processor/processor.php';

// Se há um POST, processa o texto
$result = null;
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['text'])) {
    $processor = new TextProcessor();
    $result = $processor->processText($_POST['text']);
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sistema de Processamento de Texto - Cora</title>
    <link rel="stylesheet" href="assets/style.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <div class="container">
        <header class="header">
            <h1><i class="fas fa-microphone-alt"></i> Sistema de Processamento de Texto</h1>
            <p class="subtitle">Normalização e Chunking Semântico para ChromaDB</p>
        </header>

        <main class="main-content">
            <?php if (!$result): ?>
            <!-- Formulário de Input -->
            <section class="input-section">
                <form method="POST" class="form-container">
                    <div class="form-group">
                        <label for="text" class="form-label">
                            <i class="fas fa-file-text"></i> Texto da Transcrição (10k - 20k caracteres)
                        </label>
                        <textarea 
                            id="text" 
                            name="text" 
                            class="text-input" 
                            placeholder="Cole aqui o texto transcrito das falas da Cora..."
                            required
                            minlength="100"
                            maxlength="50000"
                        ></textarea>
                        <div class="char-counter">
                            <span id="char-count">0</span> / 50.000 caracteres
                        </div>
                    </div>
                    
                    <div class="form-actions">
                        <button type="submit" class="btn-primary">
                            <i class="fas fa-cogs"></i> Processar Texto
                        </button>
                    </div>
                </form>
            </section>
            
            <?php else: ?>
            <!-- Resultados do Processamento -->
            <section class="results-section">
                <div class="results-header">
                    <h2>
                        <?php if ($result['success']): ?>
                            <i class="fas fa-check-circle text-success"></i> Processamento Concluído
                        <?php else: ?>
                            <i class="fas fa-exclamation-triangle text-error"></i> Erro no Processamento
                        <?php endif; ?>
                    </h2>
                    <button onclick="location.reload()" class="btn-secondary">
                        <i class="fas fa-arrow-left"></i> Processar Novo Texto
                    </button>
                </div>
                
                <?php if ($result['success']): ?>
                
                <!-- Estatísticas -->
                <div class="stats-container">
                    <h3><i class="fas fa-chart-bar"></i> Estatísticas</h3>
                    <div class="stats-grid">
                        <div class="stat-card">
                            <div class="stat-number"><?= $result['stats']['total_chunks'] ?></div>
                            <div class="stat-label">Total de Chunks</div>
                        </div>
                        <div class="stat-card">
                            <div class="stat-number"><?= count($result['stats']['unique_topics']) ?></div>
                            <div class="stat-label">Tópicos Únicos</div>
                        </div>
                        <div class="stat-card">
                            <div class="stat-number"><?= $result['stats']['total_words'] ?></div>
                            <div class="stat-label">Total de Palavras</div>
                        </div>
                        <div class="stat-card">
                            <div class="stat-number"><?= $result['stats']['avg_words_per_chunk'] ?></div>
                            <div class="stat-label">Média por Chunk</div>
                        </div>
                    </div>
                </div>
                
                <!-- Texto Normalizado -->
                <div class="section-container">
                    <h3><i class="fas fa-edit"></i> Texto Normalizado</h3>
                    <div class="content-box">
                        <pre class="normalized-text"><?= htmlspecialchars($result['normalized_text']) ?></pre>
                    </div>
                </div>
                
                <!-- Chunks Semânticos -->
                <div class="section-container">
                    <h3><i class="fas fa-puzzle-piece"></i> Chunks Semânticos</h3>
                    <div class="chunks-container">
                        <?php foreach ($result['chunks'] as $chunk): ?>
                        <div class="chunk-card">
                            <div class="chunk-header">
                                <span class="chunk-id">Chunk #<?= $chunk['id'] ?></span>
                                <span class="chunk-topic"><?= htmlspecialchars($chunk['topic']) ?></span>
                                <span class="cora-badge"><i class="fas fa-user"></i> Cora</span>
                            </div>
                            <div class="chunk-content">
                                <?= nl2br(htmlspecialchars($chunk['content'])) ?>
                            </div>
                            <div class="chunk-meta">
                                <span class="word-count"><?= str_word_count($chunk['content']) ?> palavras</span>
                                <?php if (!empty($chunk['keywords'])): ?>
                                <div class="keywords">
                                    <?php foreach ($chunk['keywords'] as $keyword): ?>
                                    <span class="keyword-tag"><?= htmlspecialchars($keyword) ?></span>
                                    <?php endforeach; ?>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                
                <!-- Export ChromaDB -->
                <div class="section-container">
                    <h3><i class="fas fa-download"></i> Export ChromaDB</h3>
                    <div class="export-container">
                        <button onclick="downloadChromaDBExport()" class="btn-primary">
                            <i class="fas fa-file-download"></i> Baixar JSON para ChromaDB
                        </button>
                        <p class="export-info">
                            Este arquivo JSON pode ser usado diretamente para inserir os chunks no ChromaDB.
                        </p>
                    </div>
                </div>
                
                <!-- JSON Estruturado -->
                <div class="section-container">
                    <h3><i class="fas fa-code"></i> JSON Estruturado</h3>
                    <div class="json-container">
                        <div class="json-header">
                            <span class="json-title">Dados formatados para ChromaDB</span>
                            <button onclick="copyJsonToClipboard()" class="btn-copy" title="Copiar JSON">
                                <i class="fas fa-copy"></i> Copiar
                            </button>
                        </div>
                        <div class="content-box">
                            <pre id="json-content" class="json-content"><?= htmlspecialchars(json_encode((new TextProcessor())->exportForChromaDB($result['chunks']), JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)) ?></pre>
                        </div>
                    </div>
                </div>
                
                <?php else: ?>
                <!-- Erro -->
                <div class="error-container">
                    <h3><i class="fas fa-exclamation-triangle"></i> Erro</h3>
                    <div class="error-message">
                        <?= htmlspecialchars($result['error']) ?>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Log do Processamento -->
                <div class="section-container">
                    <h3><i class="fas fa-list"></i> Log do Processamento</h3>
                    <div class="log-container">
                        <?php foreach ($result['log'] as $logEntry): ?>
                        <div class="log-entry"><?= htmlspecialchars($logEntry) ?></div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </section>
            <?php endif; ?>
        </main>
    </div>

    <script>
        // Contador de caracteres
        const textInput = document.getElementById('text');
        const charCount = document.getElementById('char-count');
        
        if (textInput && charCount) {
            textInput.addEventListener('input', function() {
                const count = this.value.length;
                charCount.textContent = count.toLocaleString();
                
                if (count < 100) {
                    charCount.style.color = '#e74c3c';
                } else if (count > 45000) {
                    charCount.style.color = '#f39c12';
                } else {
                    charCount.style.color = '#27ae60';
                }
            });
        }
        
        // Download ChromaDB Export
        function downloadChromaDBExport() {
            <?php if ($result && $result['success']): ?>
            const processor = <?php echo json_encode((new TextProcessor())->exportForChromaDB($result['chunks'])); ?>;
            
            const blob = new Blob([JSON.stringify(processor, null, 2)], {
                type: 'application/json'
            });
            
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'chromadb_chunks_' + new Date().toISOString().slice(0,10) + '.json';
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
            <?php endif; ?>
        }
        
        // Copiar JSON para clipboard
        function copyJsonToClipboard() {
            const jsonContent = document.getElementById('json-content');
            if (jsonContent) {
                navigator.clipboard.writeText(jsonContent.textContent).then(function() {
                    // Feedback visual
                    const button = document.querySelector('.btn-copy');
                    const originalText = button.innerHTML;
                    button.innerHTML = '<i class="fas fa-check"></i> Copiado!';
                    button.style.backgroundColor = '#27ae60';
                    
                    setTimeout(function() {
                        button.innerHTML = originalText;
                        button.style.backgroundColor = '';
                    }, 2000);
                }).catch(function(err) {
                    console.error('Erro ao copiar: ', err);
                    alert('Erro ao copiar JSON para clipboard');
                });
            }
        }
        
        // Adiciona loading state ao formulário
        const form = document.querySelector('form');
        if (form) {
            form.addEventListener('submit', function() {
                const button = this.querySelector('button[type="submit"]');
                button.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processando...';
                button.disabled = true;
            });
        }
    </script>
</body>
</html>