<?php
/**
 * Processador Principal - Orquestração da Aplicação
 * Arquivo: processor.php
 */

require_once 'config.php';

class TranscriptionCleaner {
    
    private $apiKey;
    private $apiUrl;
    private $model;
    private $maxTokensPerChunk;
    
    public function __construct() {
        $config = getOpenAIConfig();
        $this->apiKey = $config['api_key'];
        $this->apiUrl = $config['api_url'];
        $this->model = $config['model'];
        $this->maxTokensPerChunk = $config['max_tokens_per_chunk'];
    }
    
    /**
     * Processa o texto da transcrição com análise premium e callback para progresso
     */
    public function processTranscription($text, $progressCallback = null) {
        try {
            // Análise inicial do texto
            if ($progressCallback) {
                $progressCallback("🔍 Analisando características linguísticas do texto...");
            }
            
            $chunks = $this->splitTextIntoChunks($text);
            $processedChunks = [];
            $totalChunks = count($chunks);
            
            if ($progressCallback) {
                $progressCallback("📦 Texto dividido em $totalChunks segmentos para processamento premium...");
            }
            
            foreach ($chunks as $index => $chunk) {
                if ($progressCallback) {
                    $progressCallback("⚙️ Processamento premium do segmento " . ($index + 1) . " de $totalChunks...");
                }
                
                $processedChunk = $this->processChunk($chunk, $index, $totalChunks);
                
                if ($processedChunk) {
                    $processedChunks[] = $processedChunk;
                    
                    if ($progressCallback) {
                        $progressCallback("✅ Segmento " . ($index + 1) . " processado com qualidade premium!");
                    }
                } else {
                    throw new Exception("Erro no processamento premium do segmento " . ($index + 1));
                }
                
                // Pausa entre chunks para qualidade premium
                $config = getOpenAIConfig();
                if ($totalChunks > 1) {
                    sleep(getInterfaceConfig()['chunk_processing_delay']);
                }
            }
            
            $finalText = implode("\n\n", $processedChunks);
            
            // Revisão final premium obrigatória
            if ($progressCallback) {
                $progressCallback("🔬 Iniciando revisão final premium para otimização completa...");
            }
            $finalText = $this->finalReview($finalText);
            
            // Verificação de qualidade
            if ($progressCallback) {
                $progressCallback("🎯 Verificando qualidade e consistência do resultado...");
            }
            
            if ($progressCallback) {
                $progressCallback("🏆 Processamento PREMIUM concluído! Texto otimizado para IA personalizada.");
            }
            
            return $finalText;
            
        } catch (Exception $e) {
            if ($progressCallback) {
                $progressCallback("❌ ERRO PREMIUM: " . $e->getMessage());
            }
            return false;
        }
    }
    
    private function splitTextIntoChunks($text) {
        $words = explode(' ', $text);
        $chunks = [];
        $currentChunk = [];
        $currentLength = 0;
        
        foreach ($words as $word) {
            if ($currentLength + 1 >= ($this->maxTokensPerChunk * 1.3)) {
                $chunks[] = implode(' ', $currentChunk);
                $currentChunk = [$word];
                $currentLength = 1;
            } else {
                $currentChunk[] = $word;
                $currentLength++;
            }
        }
        
        if (!empty($currentChunk)) {
            $chunks[] = implode(' ', $currentChunk);
        }
        
        return $chunks;
    }
    
    private function processChunk($chunk, $chunkIndex, $totalChunks) {
        $systemPrompt = getSystemPrompt();
        $userPrompt = getUserPrompt($chunk, $chunkIndex, $totalChunks);
        
        $data = [
            'model' => $this->model,
            'messages' => [
                ['role' => 'system', 'content' => $systemPrompt],
                ['role' => 'user', 'content' => $userPrompt]
            ],
            'temperature' => 0.25,
            'max_tokens' => 16000  // Token máximo premium
        ];
        
        return $this->callOpenAI($data);
    }
    
    private function finalReview($text) {
        $systemPrompt = getFinalReviewPrompt();
        $userPrompt = "TEXTO COMPLETO PARA REVISÃO FINAL PREMIUM:\n\n" . $text;
        
        $data = [
            'model' => $this->model,
            'messages' => [
                ['role' => 'system', 'content' => $systemPrompt],
                ['role' => 'user', 'content' => $userPrompt]
            ],
            'temperature' => 0.2,
            'max_tokens' => 16000  // Token máximo premium para revisão
        ];
        
        return $this->callOpenAI($data);
    }
    
    private function callOpenAI($data) {
        $headers = [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $this->apiKey
        ];
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $this->apiUrl,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 1200,
            CURLOPT_SSL_VERIFYPEER => true
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        if (curl_error($ch)) {
            throw new Exception('Erro de conexão: ' . curl_error($ch));
        }
        
        curl_close($ch);
        
        if ($httpCode !== 200) {
            $errorInfo = json_decode($response, true);
            $errorMsg = isset($errorInfo['error']['message']) ? $errorInfo['error']['message'] : 'Erro HTTP: ' . $httpCode;
            throw new Exception($errorMsg);
        }
        
        $result = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('Erro ao processar resposta da API');
        }
        
        if (!isset($result['choices'][0]['message']['content'])) {
            throw new Exception('Resposta inválida da API');
        }
        
        return trim($result['choices'][0]['message']['content']);
    }
}

// PROCESSAMENTO DO FORMULÁRIO
$result = '';
$error = '';
$originalLength = 0;
$processedLength = 0;
$processingLog = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validação da chave da API
    $config = getOpenAIConfig();
    if ($config['api_key'] === 'sua-chave-api-openai-aqui' || empty($config['api_key'])) {
        $error = '🔑 ERRO: Configure sua chave da API OpenAI no arquivo config.php para acesso premium.';
    } else {
        $inputText = trim($_POST['input_text'] ?? '');
        $validationRules = getValidationRules();
        
        // Validações de entrada para processamento premium
        if (empty($inputText)) {
            $error = '📝 Texto obrigatório! Cole sua transcrição para processamento premium.';
        } elseif (strlen($inputText) < $validationRules['min_text_length']) {
            $error = '📏 Texto muito curto para análise rica. Mínimo: ' . number_format($validationRules['min_text_length']) . ' caracteres para qualidade premium.';
        } elseif (strlen($inputText) > $validationRules['max_text_length']) {
            $error = '📊 Texto excede limite atual: ' . number_format($validationRules['max_text_length']) . ' caracteres. Divida em partes ou ajuste configurações.';
        } else {
            try {
                $originalLength = strlen($inputText);
                
                // Callback para log de progresso premium
                $logCallback = function($message) use (&$processingLog) {
                    $processingLog[] = date('H:i:s') . ' - ' . $message;
                };
                
                // Log inicial
                $processingLog[] = date('H:i:s') . ' - 💎 MODO PREMIUM ATIVADO: Processamento sem limite de tokens';
                $processingLog[] = date('H:i:s') . ' - 📊 Texto original: ' . number_format($originalLength) . ' caracteres';
                
                // Processa o texto sem contexto de reunião
                $cleaner = new TranscriptionCleaner();
                $result = $cleaner->processTranscription($inputText, $logCallback);
                
                if ($result) {
                    $processedLength = strlen($result);
                    $processingLog[] = date('H:i:s') . ' - 🎉 SUCESSO: Texto enriquecido com ' . number_format($processedLength) . ' caracteres';
                } else {
                    $error = '❌ Falha no processamento premium. Verifique log detalhado para diagnóstico.';
                }
                
            } catch (Exception $e) {
                $error = '⚠️ Erro premium: ' . $e->getMessage();
            }
        }
    }
}

// Incluir o template HTML
$pageTitle = "Limpeza de Transcrições - GPT-4o-mini";
$pageData = compact('result', 'error', 'originalLength', 'processedLength', 'processingLog');

include 'template.php';
?>