<?php
/**
 * Template HTML - Interface Web
 * Arquivo: template.php
 */

$interfaceConfig = getInterfaceConfig();
$messages = getMessages();
$validationRules = getValidationRules();
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($pageTitle ?? $interfaceConfig['app_name']) ?></title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            line-height: 1.6;
            color: #333;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            border-radius: 15px;
            box-shadow: 0 20px 40px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        
        .header {
            background: linear-gradient(135deg, #4CAF50 0%, #45a049 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        
        .header h1 {
            font-size: 2.5rem;
            margin-bottom: 10px;
        }
        
        .header p {
            font-size: 1.1rem;
            opacity: 0.9;
        }
        
        .content {
            padding: 30px;
        }
        
        .form-group {
            margin-bottom: 25px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: bold;
            color: #555;
        }
        
        .form-group input,
        .form-group textarea,
        .form-group select {
            width: 100%;
            padding: 12px;
            border: 2px solid #ddd;
            border-radius: 8px;
            font-size: 14px;
            font-family: inherit;
            transition: border-color 0.3s ease;
        }
        
        .form-group input:focus,
        .form-group textarea:focus,
        .form-group select:focus {
            outline: none;
            border-color: #4CAF50;
            box-shadow: 0 0 0 3px rgba(76, 175, 80, 0.1);
        }
        
        .form-group textarea {
            resize: vertical;
            min-height: 150px;
        }
        
        .btn {
            background: linear-gradient(135deg, #4CAF50 0%, #45a049 100%);
            color: white;
            padding: 15px 30px;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: bold;
            cursor: pointer;
            transition: transform 0.2s ease, box-shadow 0.2s ease;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 20px rgba(76, 175, 80, 0.3);
        }
        
        .btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }
        
        .btn-secondary {
            background: linear-gradient(135deg, #007bff 0%, #0056b3 100%);
        }
        
        .btn-secondary:hover {
            box-shadow: 0 10px 20px rgba(0, 123, 255, 0.3);
        }
        
        .error {
            background: #ffebee;
            color: #c62828;
            padding: 15px;
            border-radius: 8px;
            border-left: 4px solid #c62828;
            margin-bottom: 20px;
        }
        
        .success {
            background: #e8f5e8;
            color: #2e7d32;
            padding: 15px;
            border-radius: 8px;
            border-left: 4px solid #4CAF50;
            margin-bottom: 20px;
        }
        
        .info {
            background: #e3f2fd;
            color: #1565c0;
            padding: 15px;
            border-radius: 8px;
            border-left: 4px solid #2196f3;
            margin-bottom: 20px;
        }
        
        .stats {
            background: #f5f5f5;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
        }
        
        .stat-item {
            text-align: center;
        }
        
        .stat-number {
            font-size: 1.5rem;
            font-weight: bold;
            color: #4CAF50;
        }
        
        .stat-label {
            font-size: 0.9rem;
            color: #666;
        }
        
        .log {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 20px;
            max-height: 200px;
            overflow-y: auto;
        }
        
        .log h4 {
            margin-bottom: 10px;
            color: #495057;
        }
        
        .log-item {
            font-size: 0.9rem;
            color: #6c757d;
            margin-bottom: 5px;
        }
        
        .result-container {
            margin-top: 30px;
        }
        
        .result-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
            flex-wrap: wrap;
            gap: 10px;
        }
        
        .result-actions {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }
        
        .copy-btn {
            background: #007bff;
            color: white;
            border: none;
            padding: 8px 15px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 14px;
        }
        
        .copy-btn:hover {
            background: #0056b3;
        }
        
        .char-count {
            font-size: 0.9rem;
            color: #666;
            margin-top: 5px;
        }
        
        .char-count.warning {
            color: #ff9800;
            font-weight: bold;
        }
        
        .char-count.error {
            color: #f44336;
            font-weight: bold;
        }
        
        .loading {
            display: none;
            text-align: center;
            padding: 20px;
        }
        
        .spinner {
            border: 4px solid #f3f3f3;
            border-top: 4px solid #4CAF50;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            margin: 0 auto 10px;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .context-suggestions {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 10px;
            margin-top: 5px;
        }
        
        .context-tag {
            display: inline-block;
            background: #e9ecef;
            color: #495057;
            padding: 4px 8px;
            margin: 2px;
            border-radius: 4px;
            font-size: 12px;
            cursor: pointer;
            transition: background-color 0.2s;
        }
        
        .context-tag:hover {
            background: #4CAF50;
            color: white;
        }
        
        @media (max-width: 768px) {
            .header h1 {
                font-size: 2rem;
            }
            
            .content {
                padding: 20px;
            }
            
            .stats {
                grid-template-columns: 1fr;
            }
            
            .result-header {
                flex-direction: column;
                align-items: flex-start;
            }
            
            .result-actions {
                width: 100%;
                justify-content: flex-start;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>💎 <?= htmlspecialchars($interfaceConfig['app_name']) ?></h1>
            <p><?= htmlspecialchars($interfaceConfig['app_subtitle']) ?></p>
        </div>
        
        <div class="content">
            <?php if (!empty($pageData['error'])): ?>
                <div class="error">
                    <strong>❌ Erro:</strong> <?= htmlspecialchars($pageData['error']) ?>
                </div>
            <?php endif; ?>
            
            <?php if (!validateConfig()): ?>
                <div class="info">
                    <strong>🔑 Configuração Premium:</strong> 
                    Para usar o processamento premium, configure sua chave da API OpenAI no arquivo <code>config.php</code>.
                    <br><br>
                    <strong>Passos:</strong>
                    <ol style="margin-left: 20px; margin-top: 10px;">
                        <li>Obtenha uma chave em <a href="https://platform.openai.com/api-keys" target="_blank">OpenAI Platform</a></li>
                        <li>Edite o arquivo <code>config.php</code></li>
                        <li>Substitua <code>'sua-chave-api-openai-aqui'</code> pela sua chave real</li>
                    </ol>
                </div>
            <?php else: ?>
                <div class="info">
                    <strong>💎 MODO PREMIUM ATIVO:</strong> 
                    Sem limite de tokens • Resposta ultra-rica • Preservação total de personalidade
                    <br>
                    <strong>🎯 Otimizado para:</strong> Embedding de IA • Personalização avançada • Máxima fidelidade linguística
                </div>
            <?php endif; ?>
            
            <form method="POST" id="processingForm">
                <div class="form-group">
                    <label for="input_text">
                        🎤 Transcrição da Pessoa <span style="color: red;">*</span>
                        <small style="font-weight: normal; color: #666;">
                            • Cole as falas que serão usadas para personalização de IA
                        </small>
                    </label>
                    <textarea 
                        id="input_text" 
                        name="input_text" 
                        placeholder="Cole aqui a transcrição completa das falas da pessoa..."
                        
                        required
                        style="height: 350px;"
                    ><?= htmlspecialchars($_POST['input_text'] ?? '') ?></textarea>
                    <div class="char-count">
                        <span id="charCount">0</span> caracteres
                        • Mín: <?= number_format($validationRules['min_text_length']) ?>
                        • Máx: <?= number_format($validationRules['max_text_length']) ?>
                        <span id="premiumIndicator" style="color: #4CAF50; font-weight: bold; margin-left: 10px;">💎 PREMIUM</span>
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn" id="submitBtn">
                        💎 Processamento Premium
                    </button>
                    <button type="button" class="btn btn-secondary" onclick="clearForm()">
                        🗑️ Limpar
                    </button>
                </div>
            </form>
            
            <div class="loading" id="loadingDiv">
                <div class="spinner"></div>
                <p>Processando sua transcrição...</p>
                <small>Isso pode levar alguns minutos dependendo do tamanho do texto.</small>
            </div>
            
            <?php if (!empty($pageData['processingLog']) && $interfaceConfig['show_processing_log']): ?>
                <div class="log">
                    <h4>📊 Log de Processamento:</h4>
                    <?php foreach ($pageData['processingLog'] as $logItem): ?>
                        <div class="log-item"><?= htmlspecialchars($logItem) ?></div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
            
            <?php if (!empty($pageData['result'])): ?>
                <div class="success">
                    <strong>🎉 Sucesso Premium:</strong> Transcrição processada com qualidade excepcional! Texto enriquecido e otimizado para IA personalizada.
                </div>
                
                <?php if ($interfaceConfig['show_statistics']): ?>
                    <div class="stats">
                        <div class="stat-item">
                            <div class="stat-number"><?= number_format($pageData['originalLength']) ?></div>
                            <div class="stat-label">Caracteres Originais</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-number" style="color: #2196F3;"><?= number_format($pageData['processedLength']) ?></div>
                            <div class="stat-label">Caracteres Premium</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-number">
                                <?php 
                                $enrichment = $pageData['originalLength'] > 0 ? (($pageData['processedLength'] - $pageData['originalLength']) / $pageData['originalLength']) * 100 : 0;
                                echo ($enrichment >= 0 ? '+' : '') . number_format($enrichment, 1) . '%';
                                ?>
                            </div>
                            <div class="stat-label">Enriquecimento</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-number" style="color: #FF9800;"><?= number_format(str_word_count($pageData['result'])) ?></div>
                            <div class="stat-label">Palavras Finais</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-number" style="color: #9C27B0;"><?= number_format(substr_count($pageData['result'], '.')) ?></div>
                            <div class="stat-label">Sentenças</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-number" style="color: #4CAF50;">⭐</div>
                            <div class="stat-label">Qualidade Premium</div>
                        </div>
                    </div>
                    
                    <div class="info">
                        <strong>🧠 Análise Premium:</strong>
                        <?php 
                        $density = $pageData['processedLength'] > 0 ? round(str_word_count($pageData['result']) / ($pageData['processedLength'] / 100), 1) : 0;
                        $avgSentenceLength = substr_count($pageData['result'], '.') > 0 ? round(str_word_count($pageData['result']) / substr_count($pageData['result'], '.'), 1) : 0;
                        ?>
                        Densidade informacional: <strong><?= $density ?> palavras/100 chars</strong> • 
                        Complexidade média: <strong><?= $avgSentenceLength ?> palavras/sentença</strong> • 
                        Otimização: <strong>Embedding-Ready</strong>
                    </div>
                <?php endif; ?>
                
                <div class="result-container">
                    <div class="result-header">
                        <h3>💎 Resultado Premium - Pronto para IA:</h3>
                        <div class="result-actions">
                            <button type="button" class="copy-btn" onclick="copyResult()">
                                📋 Copiar Texto
                            </button>
                            <button type="button" class="copy-btn" onclick="downloadResult()">
                                💾 Download Premium
                            </button>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <textarea 
                            id="result_text"
                            readonly 
                            style="height: 450px; background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%); font-family: 'Georgia', serif; line-height: 1.8;"
                        ><?= htmlspecialchars($pageData['result']) ?></textarea>
                        <div class="char-count">
                            <strong>🎯 Resultado Premium:</strong> Preservação total de personalidade • Enriquecimento contextual • Otimizado para embedding
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>
    
    <script>
        // Configurações do JavaScript
        const config = {
            minLength: <?= $validationRules['min_text_length'] ?>,
            maxLength: <?= $validationRules['max_text_length'] ?>
        };
        
        // Elementos DOM
        const inputText = document.getElementById('input_text');
        const charCount = document.getElementById('charCount');
        const submitBtn = document.getElementById('submitBtn');
        const loadingDiv = document.getElementById('loadingDiv');
        const processingForm = document.getElementById('processingForm');
        
        // Contador de caracteres
        function updateCharCount() {
            const count = inputText.value.length;
            charCount.textContent = count.toLocaleString();
            
            // Remove classes anteriores
            charCount.classList.remove('warning', 'error');
            
            if (count > config.maxLength) {
                charCount.classList.add('error');
                submitBtn.disabled = true;
            } else if (count < config.minLength) {
                charCount.classList.add('warning');
                submitBtn.disabled = count === 0;
            } else {
                charCount.classList.remove('warning', 'error');
                submitBtn.disabled = false;
            }
        }
        
        inputText.addEventListener('input', updateCharCount);
        updateCharCount();
        
        // Função para limpar formulário
        function clearForm() {
            if (confirm('Tem certeza que deseja limpar o formulário?')) {
                document.getElementById('input_text').value = '';
                updateCharCount();
            }
        }
        
        // Função para copiar resultado
        function copyResult() {
            const resultText = document.getElementById('result_text');
            resultText.select();
            document.execCommand('copy');
            
            const copyBtn = event.target;
            const originalText = copyBtn.innerHTML;
            copyBtn.innerHTML = '✅ Copiado!';
            copyBtn.style.background = '#28a745';
            
            setTimeout(() => {
                copyBtn.innerHTML = originalText;
                copyBtn.style.background = '#007bff';
            }, 2000);
        }
        
        // Função para download do resultado
        function downloadResult() {
            const resultText = document.getElementById('result_text').value;
            const blob = new Blob([resultText], { type: 'text/plain' });
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'transcricao_premium_' + new Date().toISOString().slice(0, 10) + '.txt';
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            window.URL.revokeObjectURL(url);
        }
        
        // Loading premium durante envio do formulário
        processingForm.addEventListener('submit', function() {
            submitBtn.disabled = true;
            submitBtn.innerHTML = '💎 Processando Premium...';
            loadingDiv.style.display = 'block';
            
            // Anima texto premium
            let dots = 0;
            const interval = setInterval(() => {
                dots = (dots + 1) % 4;
                submitBtn.innerHTML = '💎 Processando Premium' + '.'.repeat(dots);
            }, 500);
            
            // Para animação quando página recarregar
            window.addEventListener('beforeunload', () => {
                clearInterval(interval);
            });
        });
        
        // Restaura botão se houve erro
        window.addEventListener('load', function() {
            if (submitBtn.disabled && !inputText.value) {
                submitBtn.disabled = false;
                submitBtn.innerHTML = '💎 Processamento Premium';
                loadingDiv.style.display = 'none';
            }
        });
        
        // Auto-resize do textarea
        inputText.addEventListener('input', function() {
            this.style.height = 'auto';
            this.style.height = Math.max(300, this.scrollHeight) + 'px';
        });
    </script>
</body>
</html>