<?php
header('Content-Type: application/json');

// Lê e decodifica o JSON recebido
$data = json_decode(file_get_contents('php://input'), true);

// Validação dos campos obrigatórios, incluindo os da etapa
$requiredFields = [
    'id', 'inputName', 'titulo', 'descricao', 
    'criterioSim', 'criterioNao', 'classe', 'faixaEtaria',
    'etapa_item', 'etapa_descricao'
];

foreach ($requiredFields as $field) {
    if (!isset($data[$field]) || trim($data[$field]) === '') {
        echo json_encode(['error' => "Campo obrigatório '$field' ausente."]);
        exit;
    }
}

// Substitui texto entre aspas duplas por aspas simples
function substituirAspasInternas($texto) {
    return preg_replace('/"([^"]+)"/', "'$1'", $texto);
}

// Determina a área a partir do inputName
function getAreaFromInput($inputName) {
    if (str_starts_with($inputName, 'social')) {
        return 'Social';
    } elseif (str_starts_with($inputName, 'cognicao')) {
        return 'Cognição';
    } elseif (str_starts_with($inputName, 'linguagem')) {
        return 'Linguagem';
    } elseif (str_starts_with($inputName, 'autocuidados')) {
        return 'Autocuidados';
    } elseif (str_starts_with($inputName, 'motor')) {
        return 'Motor';
    }
    return 'Desconhecida';
}

// Formata a faixa etária
function formatFaixaEtaria($faixa) {
    $formatada = str_replace('_', '-', $faixa);
    return $formatada === '0-1' ? $formatada . ' ano' : $formatada . ' anos';
}

// Trata as orientações (até 10 itens)
$orientacoes = [];
for ($i = 1; $i <= 10; $i++) {
    $key = 'orient_' . $i;
    if (!empty(trim($data[$key] ?? ''))) {
        $orientacoes[] = substituirAspasInternas(trim($data[$key]));
    }
}

// Cria a estrutura da etapa
$etapas = [
    [
        'codigo'    => substituirAspasInternas(trim($data['etapa_item'])),
        'descricao' => substituirAspasInternas(trim($data['etapa_descricao']))
    ]
];

// Constrói a estrutura final
$estrutura = [
    'id'           => $data['id'],
    'area'         => getAreaFromInput($data['inputName']),
    'inputName'    => $data['inputName'],
    'titulo'       => substituirAspasInternas($data['id'] . ' - ' . $data['titulo']),
    'descricao'    => substituirAspasInternas($data['descricao']),
    'criterioSim'  => substituirAspasInternas($data['criterioSim']),
    'criterioNao'  => substituirAspasInternas($data['criterioNao']),
    'orientacoes'  => $orientacoes,
    'classe'       => substituirAspasInternas(preg_replace('/(?<=\))/', ' ', $data['classe'])),
    'faixaEtaria'  => formatFaixaEtaria($data['faixaEtaria']),
    'etapas'       => $etapas
];

// Gera o objeto JavaScript formatado
function gerarObjetoJavaScript($estrutura) {
    $saida = "{\n";
    foreach ($estrutura as $chave => $valor) {
        // Caso seja um array (orientacoes ou etapas)
        if (is_array($valor)) {
            $saida .= "  $chave: [\n";
            foreach ($valor as $item) {
                if (is_array($item)) {
                    $saida .= "    {\n";
                    foreach ($item as $subChave => $subValor) {
                        $saida .= "      $subChave: '" . addslashes($subValor) . "',\n";
                    }
                    $saida .= "    },\n";
                } else {
                    $saida .= "    '" . addslashes($item) . "',\n";
                }
            }
            $saida .= "  ],\n";
        } else {
            $saida .= "  $chave: '" . addslashes($valor) . "',\n";
        }
    }
    $saida .= "},\n";
    return $saida;
}

$jsFormatado = gerarObjetoJavaScript($estrutura);

// Remove escapes opcionais e espaços indesejados na barra
$jsFormatado = str_replace('\"', '"', $jsFormatado);
$jsFormatado = preg_replace('/\s*\/\s*/', '/', $jsFormatado);

// Define o nome do arquivo para salvar a saída
$arquivo = 'respostas.txt';

// Salva o resultado no arquivo utilizando FILE_APPEND
if (file_put_contents($arquivo, $jsFormatado, FILE_APPEND) !== false) {
    echo json_encode(['status' => 'Sucesso']);
} else {
    echo json_encode(['error' => 'Não foi possível escrever no arquivo.']);
}
?>
