<?php
/**
 * Script para avaliação do M-CHAT com melhorias de segurança, validação e conformidade.
 * 
 * @author
 * @version 2.0
 */

// Definir cabeçalho para UTF-8 e JSON
header('Content-Type: application/json; charset=utf-8');

// Definir fuso horário padrão
date_default_timezone_set('America/Sao_Paulo');

/**
 * Classe MChatEvaluator para avaliação do M-CHAT.
 */
class MChatEvaluator
{
    // Propriedades
    private $responses;
    private $errors = [];
    private $validResponses = [];
    private $compromisedAreas = [];
    private $atRiskItems = [];
    private $atRiskCriticalItems = [];
    private $riskScore = 0;
    private $criticalFailures = 0;
    private $riskLevel;
    private $firstName;
    private $ageInMonths;

    // Constantes
    const REVERSE_SCORED_ITEMS = ["item2", "item5", "item12"];
    const CRITICAL_ITEMS = ["item2", "item7", "item9", "item13", "item14", "item15"];
    const ITEM_AREAS = [
        "item1" => ["interação", "atenção", "comunicação verbal e não-verbal", "compreensão"],
        "item2" => ["sensibilidade e resposta auditiva", "atenção", "compreensão"],
        "item3" => ["baixo repertório de brincar simbólico", "uso de objetos", "comportamento", "interação"],
        "item4" => ["nível de atividade", "uso corporal", "comportamento"],
        "item5" => ["comportamento", "uso corporal", "atenção"],
        "item6" => ["comunicação verbal e não-verbal", "interação", "compreensão"],
        "item7" => ["comunicação verbal e não-verbal", "interação", "atenção", "compreensão"],
        "item8" => ["relações pessoais", "interação", "comportamento"],
        "item9" => ["interação", "comunicação verbal e não-verbal", "compreensão", "relações pessoais"],
        "item10" => ["sensibilidade e resposta auditiva", "atenção", "compreensão"],
        "item11" => ["interação", "relações pessoais", "comunicação verbal e não-verbal"],
        "item12" => ["sensibilidade e resposta auditiva", "comportamento", "atenção"],
        "item13" => ["uso corporal", "nível de atividade"],
        "item14" => ["interação", "comunicação verbal e não-verbal", "relações pessoais"],
        "item15" => ["imitação", "interação", "comunicação verbal e não-verbal"],
        "item16" => ["atenção", "interação", "compreensão"],
        "item17" => ["interação", "comunicação verbal e não-verbal", "relações pessoais"],
        "item18" => ["compreensão", "comunicação verbal e não-verbal", "atenção"],
        "item19" => ["interação", "atenção", "compreensão"],
        "item20" => ["nível de atividade", "uso corporal", "comportamento"]
    ];

    /**
     * Construtor da classe.
     * @param array $responses Respostas recebidas.
     */
    public function __construct($responses)
    {
        $this->responses = $responses;
    }

    /**
     * Função principal para executar a avaliação.
     * @return array Resultado da avaliação ou erros.
     */
    public function evaluate()
    {
        // Sanitização e validação das respostas
        $this->validateResponses();

        if (!empty($this->errors)) {
            // Retorna erros encontrados
            return ['errors' => $this->errors];
        }

        // Calcula a idade em meses
        try {
            $this->calculateAgeInMonths();
        } catch (Exception $e) {
            return ['errors' => ['Erro ao calcular a idade: ' . $e->getMessage()]];
        }

        // Análise das respostas
        $this->analyzeResponses();

        // Geração do relatório
        $report = $this->generateReport();

        // Retorna o relatório
        return ['report' => $report];
    }

    /**
     * Valida e sanitiza as respostas fornecidas.
     */
    private function validateResponses()
    {
        // Campos obrigatórios
        $requiredFields = array_merge(array_keys(self::ITEM_AREAS), ['nome', 'data_nascimento', 'data_avaliacao']);

        // Verificar campos ausentes
        foreach ($requiredFields as $field) {
            if (!isset($this->responses[$field]) || empty(trim($this->responses[$field]))) {
                $this->errors[] = "O campo '$field' é obrigatório.";
            }
        }

        // Se houver erros, não prosseguir
        if (!empty($this->errors)) {
            return;
        }

        // Sanitização das entradas
        foreach ($this->responses as $item => $response) {
            $response = trim($response);

            // Ignorar campos não esperados
            if (!in_array($item, $requiredFields)) {
                continue;
            }

            // Validar respostas dos itens
            if (strpos($item, 'item') === 0) {
                $response = mb_strtolower($response, 'UTF-8');
                if (!in_array($response, ["sim", "não"])) {
                    $this->errors[] = "Resposta inválida para $item: $response. Respostas válidas são 'sim' ou 'não'.";
                } else {
                    $this->validResponses[$item] = $response;
                }
            } else {
                // Sanitização de campos de texto
                $this->responses[$item] = htmlspecialchars($response, ENT_QUOTES, 'UTF-8');
            }
        }

        // Verificar se todas as perguntas foram respondidas
        $missingItems = array_diff(array_keys(self::ITEM_AREAS), array_keys($this->validResponses));

        if (!empty($missingItems)) {
            $this->errors[] = "As seguintes perguntas não foram respondidas: " . implode(", ", $missingItems) . ".";
        }
    }

    /**
     * Calcula a idade em meses da criança.
     * @throws Exception Se as datas não forem válidas.
     */
    private function calculateAgeInMonths()
    {
        $birthDate = DateTime::createFromFormat('Y-m-d', $this->responses['data_nascimento']);
        $evaluationDate = DateTime::createFromFormat('Y-m-d', $this->responses['data_avaliacao']);

        if (!$birthDate || !$evaluationDate) {
            throw new Exception('Formato de data inválido. Use o formato YYYY-MM-DD.');
        }

        $ageInterval = $birthDate->diff($evaluationDate);
        $this->ageInMonths = ($ageInterval->y * 12) + $ageInterval->m;

        if ($this->ageInMonths < 0) {
            throw new Exception('A data de avaliação não pode ser anterior à data de nascimento.');
        }

        // Extrair o primeiro nome
        $this->firstName = explode(' ', trim($this->responses['nome']))[0];
        $this->firstName = htmlspecialchars($this->firstName, ENT_QUOTES, 'UTF-8');
    }

    /**
     * Analisa as respostas válidas e calcula o risco.
     */
    private function analyzeResponses()
    {
        foreach ($this->validResponses as $item => $response) {
            $isAtRisk = false;

            if (in_array($item, self::REVERSE_SCORED_ITEMS)) {
                // Para esses itens, "sim" indica risco
                if ($response === "sim") {
                    $isAtRisk = true;
                }
            } else {
                // Para os demais itens, "não" indica risco
                if ($response === "não") {
                    $isAtRisk = true;
                }
            }

            if ($isAtRisk) {
                $this->riskScore++;
                $this->atRiskItems[] = $item;

                // Verifica se é um item crítico
                if (in_array($item, self::CRITICAL_ITEMS)) {
                    $this->atRiskCriticalItems[] = $item;
                }

                // Adiciona as áreas comprometidas, evitando duplicatas
                foreach (self::ITEM_AREAS[$item] as $area) {
                    $areaLower = mb_strtolower($area, 'UTF-8');
                    if (!in_array($areaLower, $this->compromisedAreas)) {
                        $this->compromisedAreas[] = $areaLower;
                    }
                }
            }
        }

        $this->criticalFailures = count($this->atRiskCriticalItems);
        $this->determineRiskLevel();
    }

    /**
     * Determina o nível de risco com base na pontuação.
     */
    private function determineRiskLevel()
    {
        if ($this->riskScore >= 8) {
            $this->riskLevel = "Alto Risco";
        } elseif ($this->riskScore >= 3) {
            $this->riskLevel = "Risco Médio";
        } else {
            $this->riskLevel = "Baixo Risco";
        }

        // Ajusta para Alto Risco se houver falhas em 2 ou mais itens críticos
        if ($this->criticalFailures >= 2 && $this->riskScore < 8) {
            $this->riskLevel = "Alto Risco";
        }
    }

    /**
     * Gera o relatório com base na análise.
     * @return string Relatório gerado.
     */
    private function generateReport()
    {
        // Correção de concordância para "falha(s)"
        if ($this->criticalFailures == 0) {
            $criticalFailuresText = "sem falhas em itens críticos";
        } elseif ($this->criticalFailures == 1) {
            $criticalFailuresText = "com 1 falha em itens críticos";
        } else {
            $criticalFailuresText = "com {$this->criticalFailures} falhas em itens críticos";
        }

        // Listar as áreas comprometidas em minúsculas
        $areasCount = count($this->compromisedAreas);
        if ($areasCount > 1) {
            $areasText = implode(", ", array_slice($this->compromisedAreas, 0, -1));
            $areasText .= " e/ou " . end($this->compromisedAreas);
        } elseif ($areasCount == 1) {
            $areasText = $this->compromisedAreas[0];
        } else {
            $areasText = "";
        }

        // Preparar a lista de itens que apresentaram falhas, marcando os críticos
        $itemsTextList = [];
        foreach ($this->atRiskItems as $item) {
            $itemNumber = substr($item, 4); // Extrai o número do item
            if (in_array($item, $this->atRiskCriticalItems)) {
                $itemsTextList[] = $itemNumber . " (crítico)";
            } else {
                $itemsTextList[] = $itemNumber;
            }
        }

        // Formatar a lista de itens com pontuação
        $itemsCount = count($itemsTextList);
        if ($itemsCount > 1) {
            $lastItem = array_pop($itemsTextList);
            $itemsText = implode(", ", $itemsTextList) . " e " . $lastItem;
            $itemsPhrase = ($itemsCount > 1) ? "Os itens que apresentaram falhas foram: $itemsText." : "O item que apresentou falha foi: $itemsText.";
        } elseif ($itemsCount == 1) {
            $itemsText = $itemsTextList[0];
            $itemsPhrase = "O item que apresentou falha foi: $itemsText.";
        } else {
            $itemsPhrase = "Nenhum item apresentou falha.";
        }

        // Construção do relatório
        if ($this->riskScore == 0 && $this->criticalFailures == 0) {
            $report = "Com base nas respostas fornecidas no M-CHAT R/F (Robins, Fein & Barton, 2014), {$this->firstName}, com {$this->ageInMonths} meses, não apresenta sinais de risco para Transtorno do Espectro Autista.";
        } else {
            $report = "Com base nas respostas fornecidas no M-CHAT R/F (Robins, Fein & Barton, 2014), {$this->firstName}, com {$this->ageInMonths} meses, apresenta uma pontuação de risco de {$this->riskScore}, {$criticalFailuresText}. As respostas ao instrumento indicam possíveis comprometimentos em áreas como {$areasText}. {$itemsPhrase}";

            if ($this->criticalFailures >= 2 && $this->riskScore < 8) {
                $report .= " Apesar da pontuação total ser menor que 8, houve falhas em {$this->criticalFailures} itens críticos, o que sugere um Alto Risco de Transtorno do Espectro Autista.";
            } else {
                $report .= " Isso indica um {$this->riskLevel} de Transtorno do Espectro Autista.";
            }
        }

        // Verificar se a idade está fora da faixa recomendada
        $ageObservation = "";
        if ($this->ageInMonths < 16 || $this->ageInMonths > 30) {
            $ageObservation = "Observação: {$this->firstName} está fora da faixa etária recomendada para o M-CHAT (16 a 30 meses).";
        } elseif ($this->ageInMonths < 24 && $this->riskLevel == "Baixo Risco") {
            $ageObservation = "Recomenda-se repetir o M-CHAT aos 24 meses.";
        }

        // Adicionar observação ao relatório
        if ($ageObservation) {
            $report .= " {$ageObservation}";
        }

        return $report;
    }
}

// Receber dados JSON do corpo da requisição
$inputData = file_get_contents('php://input');
$responses = json_decode($inputData, true);

// Verificar se os dados foram fornecidos e se o JSON é válido
if (json_last_error() !== JSON_ERROR_NONE || !$responses) {
    echo json_encode(['errors' => ['Dados de entrada inválidos. Por favor, forneça um JSON válido.']]);
    exit;
}

// Criar uma instância do avaliador
$evaluator = new MChatEvaluator($responses);

// Executar a avaliação
$result = $evaluator->evaluate();

// Retornar o resultado como JSON
echo json_encode($result);
?>
