<?php
// processa.php

// Configurações do Banco de Dados
$db_host = 'localhost';        // Geralmente 'localhost' no cPanel
$db_user = 'wwifcora_teste';      // Seu usuário do MySQL
$db_pass = 'Lucas@31!';        // Sua senha do MySQL
$db_name = 'wwifcora_teste';       // Nome do banco de dados


// ---------------------------------------------------------------------
// 2. FUNÇÕES AUXILIARES
// ---------------------------------------------------------------------

/**
 * Conecta ao banco de dados MySQL.
 *
 * @param string $host
 * @param string $user
 * @param string $pass
 * @param string $name
 * @return mysqli
 * @throws Exception
 */
function conectar_db($host, $user, $pass, $name) {
    $conn = new mysqli($host, $user, $pass, $name);
    if ($conn->connect_error) {
        throw new Exception("Falha na conexão com o banco de dados: " . $conn->connect_error);
    }
    return $conn;
}

/**
 * Carrega e valida o JSON recebido via POST.
 *
 * @param string $json_input
 * @return array
 * @throws Exception
 */
function carregar_respostas($json_input) {
    $respostas = json_decode($json_input, true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception("JSON inválido: " . json_last_error_msg());
    }

    // Garante que as chaves soc0_1_item1 até soc0_1_item28 existam e que o valor seja 1,2,3
    for ($i = 1; $i <= 28; $i++) {
        $chave = "soc0_1_item" . $i;
        if (!isset($respostas[$chave])) {
            throw new Exception("Falta o item '$chave' no JSON.");
        }
        if (!in_array($respostas[$chave], [1,2,3])) {
            throw new Exception("Valor inválido para '$chave': deve ser 1, 2 ou 3.");
        }
    }

    return $respostas;
}

/**
 * Obtém as classes que apresentam dificuldades (itens com valor 1 ou 2).
 *
 * @param array $respostas
 * @param mysqli $conn
 * @return array
 * @throws Exception
 */
function obter_classes_com_dificuldade($respostas, $conn) {
    $sql = "SELECT identifier, class_nome, resposta_padrao, itens FROM Classes";
    $result = $conn->query($sql);

    if (!$result) {
        throw new Exception("Erro ao consultar o banco de dados: " . $conn->error);
    }

    // Mapeia item -> (classe)
    $item_para_classe = [];
    while ($row = $result->fetch_assoc()) {
        $identifier = $row['identifier'];
        $class_nome = $row['class_nome'];
        $resposta_padrao = $row['resposta_padrao'];
        $lista_itens = explode(',', $row['itens']);
        
        // Cada classe pode ter vários itens
        foreach ($lista_itens as $item_bruto) {
            $item_bruto = trim($item_bruto);
            $item_para_classe[$item_bruto] = [
                'identifier'      => $identifier,
                'class_nome'      => $class_nome,
                'resposta_padrao' => $resposta_padrao
            ];
        }
    }

    // Classes que apresentam alguma dificuldade
    $classes_dificuldade = [];

    foreach ($respostas as $item => $valor) {
        // Valor 1 (Não) ou 2 (Às vezes) = dificuldade
        if ($valor === 1 || $valor === 2) {
            if (isset($item_para_classe[$item])) {
                $classe_info = $item_para_classe[$item];
                $key = $classe_info['identifier'];

                if (!isset($classes_dificuldade[$key])) {
                    $classes_dificuldade[$key] = [
                        'class_nome'      => $classe_info['class_nome'],
                        'resposta_padrao' => $classe_info['resposta_padrao'],
                        'itens'           => []
                    ];
                }

                // Extrai apenas o número do item (soc0_1_itemX -> X)
                preg_match('/item(\d+)/', $item, $matches);
                $num_item = $matches[1] ?? $item;

                $classes_dificuldade[$key]['itens'][] = $num_item;
            }
        }
    }

    return $classes_dificuldade;
}

/**
 * Gera um relatório textual compilado em um único parágrafo,
 * usando linguagem fluida e conjunções de ligação.
 *
 * @param array $classes_dificuldade
 * @return string
 */
function gerar_relatorio($classes_dificuldade) {
    // Se não houver dificuldades, mensagem positiva
    if (empty($classes_dificuldade)) {
        return "Nenhuma área com dificuldades identificadas. Tudo indica que o desenvolvimento socioafetivo está dentro do esperado!";
    }

    // Ordena as classes para uma saída previsível (opcional)
    ksort($classes_dificuldade);

    // Array para armazenar cada "bloco" descritivo de classe
    $partes = [];

    foreach ($classes_dificuldade as $info) {
        $class_nome = $info['class_nome'];
        $resposta_padrao = trim($info['resposta_padrao']);

        // Monta uma lista de itens, ex: "itens 8 e 23"
        $itens = $info['itens'];
        sort($itens);  // Ordena só para ficar "8, 23" em ordem crescente
        $count_itens = count($itens);

        if ($count_itens === 1) {
            // Apenas 1 item
            $itens_str = "item " . $itens[0];
        } else {
            // Ex: "itens 8 e 23" ou "itens 8, 9 e 11"
            $ultima = array_pop($itens);
            $itens_str = "itens " . implode(", ", $itens) . " e " . $ultima;
        }

        // Monta a frase para esta classe
        // Ex: "“Acariciar/abraçar” ($itens_str), pouca expressão de afeto físico"
        $frase = "“{$class_nome}” ({$itens_str}), {$resposta_padrao}";
        $partes[] = $frase;
    }

    // Agora, vamos juntar as frases em um único parágrafo,
    // usando conjunções para dar fluidez.
    // Exemplo: "..., além disso ..., também ..., e ..."
    $paragrafo = "As áreas que demandam maior atenção concentram-se em ";

    // Se tiver apenas 1 classe, retorna direto
    if (count($partes) === 1) {
        $paragrafo .= $partes[0] . ".";
        return $paragrafo;
    }

    // Se tiver mais de uma, vamos inserir conjunções
    // Ex: "frase1, além disso frase2, também frase3, e frase4"
    $conjuncoes = ["além disso", "também", "da mesma forma", "por fim", "igualmente"];
    $texto_final = "";
    $total = count($partes);

    // Percorre cada parte
    for ($i = 0; $i < $total; $i++) {
        // Se for a primeira, adiciona sem conjunção
        if ($i === 0) {
            $texto_final .= $partes[$i];
            continue;
        }
        // Se for a última, usamos "e" antes
        if ($i === $total - 1) {
            $texto_final .= ", e " . $partes[$i];
        } else {
            // Pega uma conjunção "aleatória" ou sequencial
            $cj = $conjuncoes[($i - 1) % count($conjuncoes)];
            $texto_final .= ", $cj " . $partes[$i];
        }
    }

    $paragrafo .= $texto_final . ".";

    return $paragrafo;
}

// ---------------------------------------------------------------------
// 3. LÓGICA PRINCIPAL DE PROCESSAMENTO
// ---------------------------------------------------------------------

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Recebe o JSON do campo 'json_respostas'
    $json_input = isset($_POST['json_respostas']) ? $_POST['json_respostas'] : '';

    try {
        // Carrega e valida o JSON
        $respostas = carregar_respostas($json_input);

        // Conecta ao banco de dados
        $conn = conectar_db($db_host, $db_user, $db_pass, $db_name);

        // Verifica quais classes têm dificuldade
        $classes_dificuldade = obter_classes_com_dificuldade($respostas, $conn);

        // Fecha a conexão
        $conn->close();

        // Gera o relatório formatado em parágrafo único
        $relatorio = gerar_relatorio($classes_dificuldade);

        // Exibe o relatório
        // Convertendo para HTML seguro e permitindo quebras de linha se quiser
        echo nl2br(htmlspecialchars($relatorio));

    } catch (Exception $e) {
        // Em caso de erro, exibe a mensagem
        echo "Erro: " . htmlspecialchars($e->getMessage());
    }
} else {
    // Se não for POST, mostramos mensagem ou redirecionamos
    echo "Acesso inválido. Envie os dados via POST.";
}