<?php
/**
 * Script de análise de respostas cognitivas com retorno JSON estruturado.
 */

header('Content-Type: application/json');

// ===================== CONFIGURAÇÕES =====================
$EXPECTED_API_KEY = 'sua_chave_secreta_aqui';

// (mesmos mapeamentos e respostas padrão do script original)
$item_mapping = [
    // Classe 1: Imitação Verbal
    "ling_item1" => ["class" => 1, "age_range" => "0-1"], 
    "ling_item2" => ["class" => 1, "age_range" => "0-1"], 
    "ling_item8" => ["class" => 1, "age_range" => "0-1"], 
    
    // Classe 2: Emissão de Gestos
    "ling_item3" => ["class" => 2, "age_range" => "0-1"], 
    "ling_item6" => ["class" => 2, "age_range" => "0-1"], 
    
    // Classe 3: Seguir Ordens
    "ling_item4" => ["class" => 3, "age_range" => "0-1"], 
    "ling_item5" => ["class" => 3, "age_range" => "0-1"], 
    "ling_item14" => ["class" => 3, "age_range" => "1-2"], 
    "ling_item43" => ["class" => 3, "age_range" => "2-3"], 
    "ling_item61" => ["class" => 3, "age_range" => "3-4"], 
    "ling_item71" => ["class" => 3, "age_range" => "4-5"], 
    
    // Classe 4: Produção de Sons
    "ling_item7" => ["class" => 4, "age_range" => "0-1"], 
    "ling_item10" => ["class" => 4, "age_range" => "0-1"], 
    
    // Classe 5: Nomeação de Objetos/Pessoas e Eventos
    "ling_item9" => ["class" => 5, "age_range" => "0-1"], 
    "ling_item11" => ["class" => 5, "age_range" => "1-2"], 
    "ling_item13" => ["class" => 5, "age_range" => "1-2"], 
    "ling_item22" => ["class" => 5, "age_range" => "1-2"], 
    "ling_item23" => ["class" => 5, "age_range" => "1-2"], 
    "ling_item24" => ["class" => 5, "age_range" => "1-2"], 
    "ling_item37" => ["class" => 5, "age_range" => "2-3"], 
    "ling_item56" => ["class" => 5, "age_range" => "2-3"], 
    "ling_item58" => ["class" => 5, "age_range" => "2-3"], 
    "ling_item79" => ["class" => 5, "age_range" => "4-5"], 
    "ling_item82" => ["class" => 5, "age_range" => "4-5"], 
    
    // Classe 6: Pedir/Expressar Necessidades/Vontades
    "ling_item12" => ["class" => 6, "age_range" => "1-2"], 
    "ling_item15" => ["class" => 6, "age_range" => "1-2"], 
    "ling_item21" => ["class" => 6, "age_range" => "1-2"], 
    "ling_item25" => ["class" => 6, "age_range" => "1-2"], 
    "ling_item31" => ["class" => 6, "age_range" => "2-3"], 
    
    // Classe 7: Apontar para Objetos/Figuras
    "ling_item16" => ["class" => 7, "age_range" => "1-2"], 
    "ling_item17" => ["class" => 7, "age_range" => "1-2"], 
    "ling_item40" => ["class" => 7, "age_range" => "2-3"], 
    "ling_item52" => ["class" => 7, "age_range" => "3-4"], 
    "ling_item78" => ["class" => 7, "age_range" => "4-5"], 
    
    // Classe 8: Identificação de Partes do Corpo
    "ling_item18" => ["class" => 8, "age_range" => "1-2"], 
    "ling_item27" => ["class" => 8, "age_range" => "1-2"], 
    
    // Classe 9: Auto Identificação
    "ling_item19" => ["class" => 9, "age_range" => "1-2"], 
    "ling_item39" => ["class" => 9, "age_range" => "2-3"], 
    "ling_item41" => ["class" => 9, "age_range" => "2-3"], 
    "ling_item42" => ["class" => 9, "age_range" => "2-3"], 
    "ling_item51" => ["class" => 9, "age_range" => "2-3"], 
    "ling_item62" => ["class" => 9, "age_range" => "3-4"], 
    "ling_item87" => ["class" => 9, "age_range" => "5-6"], 
    "ling_item88" => ["class" => 9, "age_range" => "5-6"], 
    
    // Classe 10: Responder Perguntas
    "ling_item20" => ["class" => 10, "age_range" => "1-2"], 
    "ling_item28" => ["class" => 10, "age_range" => "1-2"], 
    "ling_item35" => ["class" => 10, "age_range" => "2-3"], 
    "ling_item36" => ["class" => 10, "age_range" => "2-3"], 
    "ling_item53" => ["class" => 10, "age_range" => "2-3"], 
    "ling_item63" => ["class" => 10, "age_range" => "3-4"], 
    "ling_item66" => ["class" => 10, "age_range" => "3-4"], 
    "ling_item73" => ["class" => 10, "age_range" => "4-5"], 
    "ling_item76" => ["class" => 10, "age_range" => "4-5"], 
    "ling_item85" => ["class" => 10, "age_range" => "4-5"], 
    "ling_item86" => ["class" => 10, "age_range" => "5-6"], 
    "ling_item89" => ["class" => 10, "age_range" => "5-6"], 
    "ling_item93" => ["class" => 10, "age_range" => "5-6"], 
    "ling_item96" => ["class" => 10, "age_range" => "5-6"], 
    "ling_item97" => ["class" => 10, "age_range" => "5-6"], 
    
    // Classe 11: Fazer Perguntas
    "ling_item26" => ["class" => 11, "age_range" => "1-2"], 
    "ling_item47" => ["class" => 11, "age_range" => "2-3"], 
    "ling_item99" => ["class" => 11, "age_range" => "5-6"], 
    
    // Classe 12: Elaboração de Frases
    "ling_item29" => ["class" => 12, "age_range" => "2-3"], 
    "ling_item30" => ["class" => 12, "age_range" => "2-3"], 
    "ling_item32" => ["class" => 12, "age_range" => "2-3"], 
    "ling_item33" => ["class" => 12, "age_range" => "2-3"], 
    "ling_item75" => ["class" => 12, "age_range" => "4-5"], 
    "ling_item84" => ["class" => 12, "age_range" => "4-5"], 
    
    // Classe 13: Uso do Negativo
    "ling_item34" => ["class" => 13, "age_range" => "2-3"], 
    
    // Classe 14: Uso do Plural
    "ling_item38" => ["class" => 14, "age_range" => "2-3"], 
    "ling_item45" => ["class" => 14, "age_range" => "2-3"], 
    
    // Classe 15: Utilização de Verbos
    "ling_item44" => ["class" => 15, "age_range" => "2-3"], 
    "ling_item46" => ["class" => 15, "age_range" => "2-3"], 
    "ling_item50" => ["class" => 15, "age_range" => "2-3"], 
    "ling_item57" => ["class" => 15, "age_range" => "2-3"], 
    "ling_item64" => ["class" => 15, "age_range" => "3-4"], 
    "ling_item67" => ["class" => 15, "age_range" => "3-4"], 
    "ling_item69" => ["class" => 15, "age_range" => "3-4"], 
    "ling_item72" => ["class" => 15, "age_range" => "4-5"], 
    "ling_item74" => ["class" => 15, "age_range" => "4-5"], 
    "ling_item77" => ["class" => 15, "age_range" => "4-5"], 
    
    // Classe 16: Controle da Voz
    "ling_item48" => ["class" => 16, "age_range" => "2-3"], 
    
    // Classe 17: Uso do Possessivo
    "ling_item54" => ["class" => 17, "age_range" => "2-3"], 
    
    // Classe 18: Uso de Artigo
    "ling_item49" => ["class" => 18, "age_range" => "2-3"], 
    "ling_item55" => ["class" => 18, "age_range" => "2-3"], 
    
    // Classe 19: Uso de Diminutivo e Aumentativo
    "ling_item59" => ["class" => 19, "age_range" => "3-4"], 
    
    // Classe 20: Prestar Atenção
    "ling_item60" => ["class" => 20, "age_range" => "3-4"], 
    
    // Classe 21: Narrar Fatos
    "ling_item65" => ["class" => 21, "age_range" => "3-4"], 
    "ling_item70" => ["class" => 21, "age_range" => "3-4"], 
    "ling_item81" => ["class" => 21, "age_range" => "4-5"], 
    "ling_item90" => ["class" => 21, "age_range" => "5-6"], 
    "ling_item91" => ["class" => 21, "age_range" => "5-6"], 
    "ling_item92" => ["class" => 21, "age_range" => "5-6"], 
    "ling_item94" => ["class" => 21, "age_range" => "5-6"], 
    
    // Classe 22: Emprego de Masculino/Feminino
    "ling_item68" => ["class" => 22, "age_range" => "3-4"], 
    
    // Classe 23: Uso de Antônimos
    "ling_item80" => ["class" => 23, "age_range" => "4-5"], 
    
    // Classe 24: Discriminação de Rimas
    "ling_item83" => ["class" => 24, "age_range" => "4-5"], 
    
    // Classe 25: Definição de Palavras
    "ling_item95" => ["class" => 25, "age_range" => "5-6"], 
    
    // Classe 26: Noção de Tempo, Passado e Futuro
    "ling_item98" => ["class" => 26, "age_range" => "5-6"],
];
$standard_responses = [
    // Classe 1: Imitação Verbal
    1 => [
        "0-1" => "Não repetiu sons ou entonações, sugerindo atraso na imitação vocal e na aquisição inicial da fala.",
    ],
    
    // Classe 2: Emissão de Gestos
    2 => [
        "0-1" => "Não respondeu com gestos ou sinais não verbais, indicando dificuldade na comunicação gestual básica.",
    ],
    
    // Classe 3: Seguir Ordens
    3 => [
        "0-1" => "Dificuldade em obedecer ordens simples, mesmo com gestos ou proibições, mostrando atraso na compreensão inicial.",
        "1-2" => "Não atendeu ordens sem gesto indicador, indicando dificuldade na compreensão verbal.",
        "2-3" => "Falhou ao cumprir sequência de duas orientações, revelando limitação de compreensão e memória de curto prazo.",
        "3-4" => "Teve dificuldade em obedecer a 2 ordens não relacionadas, prejudicando a atenção e compreensão.",
        "4-5" => "Não seguiu 3 ordens em sequência, evidenciando atraso na habilidade de processar instruções complexas.",
    ],
    
    // Classe 4: Produção de Sons
    4 => [
        "0-1" => "Não combinou sílabas nem vocalizou em resposta à fala, indicando atraso na emissão de sons básicos.",
    ],
    
    // Classe 5: Nomeação de Objetos/Pessoas/Eventos
    5 => [
        "0-1" => "Não usou uma palavra funcionalmente para indicar objetos, sinalizando atraso na associação nome-objeto.",
        "1-2" => "Não nomeou pessoas, membros da família ou brinquedos, sugerindo vocabulário limitado.",
        "2-3" => "Falhou em nomear sons ou especificar objetos abertos/fechados, mostrando dificuldade de ampliar o vocabulário.",
        "4-5" => "Não empregou termos para parentes próximos ou apontou objetos fora de categoria, prejudicando a precisão na nomeação.",
    ],
    
    // Classe 6: Pedir/Expressar Necessidades/Vontades
    6 => [
        "1-2" => "Não pediu “mais”, não entregou objetos quando solicitado, evidenciando comunicação limitada de necessidades.",
        "2-3" => "Não usou palavra para ir ao banheiro, indicando dificuldade em expressar necessidades básicas verbalmente.",
    ],
    
    // Classe 7: Apontar para Objetos/Figuras
    7 => [
        "1-2" => "Não apontou para objetos ou figuras nomeadas, sugerindo atraso na compreensão e associação palavra-imagem.",
        "2-3" => "Falhou ao dizer que algo 'não é' outra coisa, indicando dificuldade em discriminar e rotular objetos.",
        "4-5" => "Não identificou absurdos em figuras, prejudicando a percepção de detalhes e vocabulário descritivo.",
    ],
    
    // Classe 8: Identificação de Partes do Corpo
    8 => [
        "1-2" => "Não apontou 3 partes do próprio corpo ou de bonecas, demonstrando atraso na compreensão do esquema corporal.",
    ],
    
    // Classe 9: Autoidentificação
    9 => [
        "1-2" => "Não disse seu nome ou apelido quando solicitado, sugerindo atraso na consciência de si.",
        "2-3" => "Falhou ao usar o próprio nome ao falar, revelando confusão de identidade pessoal.",
        "3-4" => "Não disse o nome completo, evidenciando dificuldade em informações pessoais básicas.",
        "5-6" => "Não lembrou endereço ou telefone, comprometendo a segurança e autonomia.",
    ],
    
    // Classe 10: Responder Perguntas
    10 => [
        "1-2" => "Não reagiu adequadamente a 'O que é isto?' ou perguntas de sim/não, indicando compreensão limitada.",
        "2-3" => "Falhou ao responder 'O que está fazendo?' ou 'Onde está?', apontando atraso no entendimento de questões simples.",
        "3-4" => "Não respondeu 'Como...' ou 'Quem...', demonstrando dificuldade em explicar usos ou identificar pessoas.",
        // Observação: O texto une 4-5 e 5-6, mas podemos subdividir para melhor organização, caso desejado
        "4-5" => "Teve dificuldade em justificar 'Por que?' ou dizer antônimos, prejudicando raciocínio e vocabulário mais complexo.",
        "5-6" => "Teve dificuldade em justificar 'Por que?' ou dizer antônimos, prejudicando raciocínio e vocabulário mais complexo.",
    ],
    
    // Classe 11: Fazer Perguntas
    11 => [
        "1-2" => "Não fez perguntas variando entonação, sugerindo atraso em iniciativa comunicativa.",
        "2-3" => "Falhou em perguntar 'O que é isso?' regularmente, indicando pouca curiosidade verbal.",
        "5-6" => "Não questionou o significado de palavras novas, prejudicando a expansão do vocabulário.",
    ],
    
    // Classe 12: Elaboração de Frases
    12 => [
        "2-3" => "Não formou frases de 2 palavras (substantivo + verbo), mostrando atraso na construção básica de sentenças.",
        "4-5" => "Dificuldade em usar orações compostas, evidenciando limitação na ampliação da estrutura frasal.",
    ],
    
    // Classe 13: Uso do Negativo
    13 => [
        "2-3" => "Não empregou 'não' na fala, indicando falha em expressar recusa ou contrariedade verbalmente.",
    ],
    
    // Classe 14: Uso do Plural
    14 => [
        "2-3" => "Não compreendeu a forma plural ou não a utilizou, revelando atraso na noção de quantidade em palavras.",
    ],
    
    // Classe 15: Utilização de Verbos
    15 => [
        "2-3" => "Não usou verbo no gerúndio ou formas irregulares, apontando limitação na flexão verbal inicial.",
        // Observação: O texto agrega 3-4 / 4-5 anos
        "3-4" => "Dificuldade em usar verbos no passado/futuro ou condicional, prejudicando a expressão temporal.",
        "4-5" => "Dificuldade em usar verbos no passado/futuro ou condicional, prejudicando a expressão temporal.",
    ],
    
    // Classe 16: Controle da Voz
    16 => [
        "2-3" => "Falhou ao regular o volume da voz, indicando pouca habilidade de modulação vocal.",
    ],
    
    // Classe 17: Uso do Possessivo
    17 => [
        "2-3" => "Não empregou formas possessivas ('meu', 'do papai'), mostrando atraso na referência de posse.",
    ],
    
    // Classe 18: Uso de Artigo
    18 => [
        "2-3" => "Não usou 'este/esta' ou artigos, apontando falha na coesão da frase.",
    ],
    
    // Classe 19: Uso de Diminutivo e Aumentativo
    19 => [
        "3-4" => "Falhou em usar 'inho / ão' adequadamente, evidenciando vocabulário limitado e falta de flexibilidade lexical.",
    ],
    
    // Classe 20: Prestar Atenção
    20 => [
        "3-4" => "Não se manteve atento por 5 minutos a uma história, mostrando baixa concentração auditiva.",
    ],
    
    // Classe 21: Narrar Fatos
    21 => [
        "3-4" => "Não relatou experiências imediatas ou fatos em sequência, indicando dificuldade em organizar a fala.",
        // Observação: O texto agrega 4-5 / 5-6 anos
        "4-5" => "Teve problemas para contar histórias ou relatar eventos diários, prejudicando a construção narrativa.",
        "5-6" => "Teve problemas para contar histórias ou relatar eventos diários, prejudicando a construção narrativa.",
    ],
    
    // Classe 22: Emprego de Masculino/Feminino
    22 => [
        "3-4" => "Não distinguiu entre masculino/feminino na fala, sugerindo atraso na concordância de gênero.",
    ],
    
    // Classe 23: Uso de Antônimos
    23 => [
        "4-5" => "Não completou frases com antônimos, revelando vocabulário restrito e dificuldade em conceitos opostos.",
    ],
    
    // Classe 24: Discriminação de Rimas
    24 => [
        "4-5" => "Não disse se duas palavras rimam, indicando atraso na percepção fonológica.",
    ],
    
    // Classe 25: Definição de Palavras
    25 => [
        "5-6" => "Não soube definir termos comuns, indicando limitação na compreensão e no vocabulário descritivo.",
    ],
    
    // Classe 26: Noção de Tempo (Passado e Futuro)
    26 => [
        "5-6" => "Não usou 'ontem' e 'amanhã' corretamente, evidenciando falha na organização temporal da fala.",
    ],
];

// ===================== FUNÇÕES DE SUPORTE =====================
function getRequestHeaders() {
    if (function_exists('getallheaders')) {
        return getallheaders();
    }
    $headers = [];
    foreach ($_SERVER as $name => $value) {
        if (substr($name, 0, 5) === 'HTTP_') {
            $headerName = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($name, 5)))));
            $headers[$headerName] = $value;
        }
    }
    return $headers;
}

function send_json_response($data, $status_code = 200) {
    http_response_code($status_code);
    echo json_encode($data);
    exit;
}

function validate_authentication($expected_key) {
    $headers = getRequestHeaders();
    if (!isset($headers['Authorization'])) {
        send_json_response(['status' => 'error', 'message' => 'Unauthorized'], 401);
    }
    if (preg_match('/Bearer\s(\S+)/', $headers['Authorization'], $matches)) {
        $provided_key = $matches[1];
    } else {
        send_json_response(['status' => 'error', 'message' => 'Unauthorized'], 401);
    }
    if ($provided_key !== $expected_key) {
        send_json_response(['status' => 'error', 'message' => 'Unauthorized'], 401);
    }
}

function get_age_order($age_range) {
    // Retorna a "prioridade" da faixa etária de acordo com o primeiro número
    $parts = explode('-', $age_range);
    return (int)$parts[0];
}

// ===================== FUNÇÃO PRINCIPAL: generate_detailed_report =====================
function generate_detailed_report($input, $item_mapping, $standard_responses) {
    $class_failures = [];

    // Percorre cada item do JSON e armazena a classe e a menor faixa etária
    foreach ($input as $item => $value) {
        // '1' significa que houve falha no item
        if ($value == 1 && isset($item_mapping[$item])) {
            $class = $item_mapping[$item]['class'];
            $age_range = $item_mapping[$item]['age_range'];
            $age_order = get_age_order($age_range);

            // Salva a menor faixa etária encontrada para aquela classe
            if (!isset($class_failures[$class]) || $age_order < $class_failures[$class]['age_order']) {
                $class_failures[$class] = [
                    'age_range' => $age_range,
                    'age_order' => $age_order
                ];
            }
        }
    }

    // Monta lista detalhada de falhas
    $failures_list = [];
    foreach ($class_failures as $class => $info) {
        $age_range = $info['age_range'];

        if (isset($standard_responses[$class][$age_range])) {
            $description = $standard_responses[$class][$age_range];
            $failures_list[] = [
                'class' => $class,
                'age_range' => $age_range,
                'description' => $description
            ];
        }
    }

    return $failures_list;
}

// ===================== EXECUÇÃO =====================

// 1. Valida a autenticação
validate_authentication($EXPECTED_API_KEY);

// 2. Lê e decodifica o corpo JSON
$json_input = file_get_contents('php://input');
$input_data = json_decode($json_input, true);

if ($input_data === null) {
    send_json_response([
        'status' => 'error',
        'message' => 'JSON inválido.'
    ], 400);
}

// 3. Gera relatório detalhado
$failures = generate_detailed_report($input_data, $item_mapping, $standard_responses);

// 4. Retorna JSON mais organizado
if (empty($failures)) {
    send_json_response([
        'status' => 'success',
        'message' => 'Nenhuma falha identificada nos itens avaliados.',
        'data' => [
            'failures_found' => [],
            'recommendations' => 'Parabéns, nenhum indicador de atraso encontrado!'
        ]
    ]);
} else {
    // Você pode, por exemplo, criar lógica para gerar recomendações baseadas nas classes identificadas
    $recommendations = 'Considere práticas de estímulo de linguagem adequadas às faixas etárias detectadas.';

    send_json_response([
        'status' => 'success',
        'message' => 'Análise realizada com sucesso.',
        'data' => [
            'failures_found' => $failures,
            'recommendations' => $recommendations
        ]
    ]);
}

