# Fix SQLite version
import sys
try:
    import pysqlite3.dbapi2 as sqlite3
    sys.modules['sqlite3'] = sqlite3
    print(f"Using SQLite version: {sqlite3.sqlite_version}")
except ImportError:
    import sqlite3
    print(f"Using system SQLite version: {sqlite3.sqlite_version}")

from fastapi import FastAPI, HTTPException, Depends, Body
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials
from fastapi.middleware.cors import CORSMiddleware
from pydantic import BaseModel
import chromadb
from chromadb.config import Settings
import os
from typing import List, Dict, Any, Optional
import openai
from dotenv import load_dotenv
import uuid

# Carregar variáveis de ambiente
load_dotenv()

app = FastAPI(title="ChromaDB API with OpenAI Embeddings", version="2.0.0")

# Adicionar CORS
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# Configurações
API_TOKEN = os.getenv("CHROMADB_API_TOKEN", "chromadb_2024_secure_token_ifcora")
OPENAI_API_KEY = os.getenv("OPENAI_API_KEY", "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")

# Configurar cliente OpenAI
client = openai.OpenAI(api_key=OPENAI_API_KEY)

# Esquema de segurança
security = HTTPBearer()

def verify_token(credentials: HTTPAuthorizationCredentials = Depends(security)):
    if credentials.credentials != API_TOKEN:
        raise HTTPException(status_code=401, detail="Token inválido")
    return credentials.credentials

def get_embeddings(texts: List[str]) -> List[List[float]]:
    """Gera embeddings usando OpenAI"""
    try:
        response = client.embeddings.create(
            model="text-embedding-3-small",
            input=texts
        )
        return [item.embedding for item in response.data]
    except Exception as e:
        print(f"Erro OpenAI: {e}")
        raise HTTPException(status_code=500, detail=f"Erro OpenAI: {str(e)}")

# Configuração do ChromaDB
chroma_client = chromadb.PersistentClient(
    path="./chroma_db",
    settings=Settings(anonymized_telemetry=False)
)

# Modelos
class DocumentModel(BaseModel):
    documents: List[str]
    metadatas: List[Dict[str, Any]] = None
    ids: List[str] = None

class QueryModel(BaseModel):
    query_texts: List[str]
    n_results: int = 10

# ENDPOINTS

@app.get("/")
async def root():
    """Endpoint público - página inicial"""
    return {
        "message": "ChromaDB API com OpenAI Embeddings funcionando!", 
        "version": "2.0.0",
        "embedding_model": "text-embedding-3-small",
        "sqlite_version": sqlite3.sqlite_version,
        "status": "healthy",
        "endpoints": {
            "health": "/health",
            "collections": "/collections",
            "docs": "/docs",
            "openapi": "/openapi.json"
        }
    }

@app.get("/health")
async def health_check():
    """Health check público"""
    return {
        "status": "healthy", 
        "service": "chromadb-api",
        "embedding_model": "text-embedding-3-small",
        "openai_configured": bool(OPENAI_API_KEY and len(OPENAI_API_KEY) > 10)
    }

@app.post("/collections/{collection_name}/add")
async def add_documents(collection_name: str, data: DocumentModel, token: str = Depends(verify_token)):
    """Adiciona documentos à coleção com embeddings automáticos"""
    try:
        print(f"Adicionando {len(data.documents)} documentos à coleção {collection_name}")
        
        # Gerar embeddings
        embeddings = get_embeddings(data.documents)
        print(f"Embeddings gerados: {len(embeddings)} vetores")
        
        # Criar ou obter coleção
        collection = chroma_client.get_or_create_collection(name=collection_name)
        
        # IDs automáticos únicos se não fornecidos
        ids = data.ids or [str(uuid.uuid4()) for _ in data.documents]
        
        # Adicionar documentos com embeddings
        collection.add(
            documents=data.documents,
            embeddings=embeddings,
            metadatas=data.metadatas,
            ids=ids
        )
        
        return {
            "message": f"Documentos adicionados à coleção {collection_name}",
            "count": len(data.documents),
            "embedding_model": "text-embedding-3-small"
        }
    except Exception as e:
        print(f"Erro ao adicionar documentos: {e}")
        raise HTTPException(status_code=500, detail=str(e))

@app.post("/collections/{collection_name}/query")
async def query_documents(collection_name: str, query: QueryModel, token: str = Depends(verify_token)):
    """Consulta documentos por similaridade semântica"""
    try:
        print(f"Consultando coleção {collection_name} com query: {query.query_texts}")
        
        # Verificar se coleção existe
        try:
            collection = chroma_client.get_collection(name=collection_name)
        except Exception:
            raise HTTPException(status_code=404, detail=f"Coleção '{collection_name}' não encontrada")
        
        # Gerar embeddings da query
        query_embeddings = get_embeddings(query.query_texts)
        print(f"Query embeddings gerados: {len(query_embeddings)} vetores")
        
        # Consultar
        results = collection.query(
            query_embeddings=query_embeddings,
            n_results=query.n_results
        )
        
        return {
            "results": results,
            "query_count": len(query.query_texts),
            "embedding_model": "text-embedding-3-small"
        }
    except HTTPException:
        raise
    except Exception as e:
        print(f"Erro ao consultar: {e}")
        raise HTTPException(status_code=500, detail=str(e))

@app.get("/collections/{collection_name}/info")
async def collection_info(collection_name: str, token: str = Depends(verify_token)):
    """Informações sobre uma coleção"""
    try:
        collection = chroma_client.get_collection(name=collection_name)
        count = collection.count()
        
        return {
            "name": collection_name,
            "count": count,
            "embedding_model": "text-embedding-3-small"
        }
    except Exception:
        raise HTTPException(status_code=404, detail=f"Coleção '{collection_name}' não encontrada")

@app.get("/collections")
async def list_collections(token: str = Depends(verify_token)):
    """Lista todas as coleções"""
    try:
        collections = chroma_client.list_collections()
        collection_info = []
        
        for col in collections:
            try:
                count = col.count()
                collection_info.append({
                    "name": col.name,
                    "count": count
                })
            except:
                collection_info.append({
                    "name": col.name,
                    "count": "unknown"
                })
        
        return {
            "collections": collection_info,
            "total_collections": len(collections),
            "embedding_model": "text-embedding-3-small"
        }
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

@app.delete("/collections/{collection_name}")
async def delete_collection(collection_name: str, token: str = Depends(verify_token)):
    """Deleta uma coleção"""
    try:
        chroma_client.delete_collection(name=collection_name)
        return {
            "message": f"Coleção '{collection_name}' deletada com sucesso"
        }
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

@app.get("/collections/{collection_name}/documents")
async def list_documents(collection_name: str, limit: int = 100, token: str = Depends(verify_token)):
    """Lista documentos de uma coleção com seus IDs"""
    try:
        collection = chroma_client.get_collection(name=collection_name)
        results = collection.get(limit=limit, include=['documents', 'metadatas'])
        
        return {
            "collection": collection_name,
            "count": len(results['ids']),
            "documents": [
                {
                    "id": results['ids'][i],
                    "document": results['documents'][i],
                    "metadata": results['metadatas'][i] if results['metadatas'] else None
                }
                for i in range(len(results['ids']))
            ]
        }
    except Exception as e:
        raise HTTPException(status_code=404, detail=f"Coleção '{collection_name}' não encontrada")
        

@app.delete("/collections/{collection_name}/documents")
async def delete_documents_by_ids(collection_name: str, document_ids: List[str] = Body(...), token: str = Depends(verify_token)):
    """Deleta documentos específicos por ID"""
    try:
        collection = chroma_client.get_collection(name=collection_name)
        collection.delete(ids=document_ids)
        
        return {
            "message": f"Deletados {len(document_ids)} documentos da coleção {collection_name}",
            "deleted_ids": document_ids
        }
    except Exception as e:
        raise HTTPException(status_code=500, detail=str(e))

if __name__ == "__main__":
    import uvicorn
    uvicorn.run(app, host="0.0.0.0", port=8000)