#!/usr/bin/env python3
"""
Teste de diagnóstico ChromaDB - versão corrigida para ambos os modos
"""

# IMPORTANTE: Fix para SQLite - deve vir ANTES de importar chromadb
import sys
import os
import pysqlite3
sys.modules['sqlite3'] = pysqlite3

print("🔧 SQLite fix aplicado - pysqlite3 substituindo sqlite3 do sistema")

try:
    import chromadb
    from chromadb.config import Settings
    print("✅ ChromaDB importado com sucesso!")
except Exception as e:
    print(f"❌ Erro ao importar ChromaDB: {e}")
    sys.exit(1)

def teste_client_local(collection_name="EC_IFCora-1"):
    """
    Teste usando PersistentClient (mesmo que a API usa)
    """
    print("\n🏠 TESTANDO CLIENT LOCAL (PersistentClient)")
    print("-" * 50)
    
    try:
        # Usar mesma configuração da API
        chroma_settings = Settings(
            anonymized_telemetry=False,
            is_persistent=True,
        )
        
        client = chromadb.PersistentClient(
            path="./chroma_db",
            settings=chroma_settings
        )
        print(f"✅ PersistentClient conectado")
        
        # Listar coleções
        collections = client.list_collections()
        collection_names = [col.name for col in collections]
        print(f"📚 Coleções disponíveis: {collection_names}")
        
        if not collection_names:
            print("ℹ️  Nenhuma coleção encontrada - banco vazio")
            return True
        
        if collection_name not in collection_names:
            print(f"❌ Coleção '{collection_name}' não encontrada!")
            print(f"   Coleções disponíveis: {collection_names}")
            return False
        
        # Obter coleção
        collection = client.get_collection(name=collection_name)
        print(f"✅ Coleção '{collection_name}' obtida")
        
        # Count oficial
        count = collection.count()
        print(f"📊 Count oficial da coleção: {count}")
        
        if count == 0:
            print("ℹ️  Coleção está vazia")
            return True
        
        # Testar diferentes limites
        print(f"\n🔍 Testando consultas com diferentes limites:")
        
        limites_teste = [5, 10, 12, 15, 20, min(count, 50)]
        
        for limite in limites_teste:
            try:
                resultado = collection.get(limit=limite)
                docs_retornados = len(resultado['ids'])
                esperado = min(limite, count)
                status = "✅" if docs_retornados == esperado else "⚠️"
                print(f"  {status} Limit {limite}: {docs_retornados}/{esperado} documentos")
                
            except Exception as e:
                print(f"  ❌ Erro com limit {limite}: {e}")
        
        # Teste sem limite
        print(f"\n🔍 Teste sem limite explícito:")
        try:
            resultado_sem_limite = collection.get()
            docs_sem_limite = len(resultado_sem_limite['ids'])
            print(f"  📄 Documentos retornados sem limite: {docs_sem_limite}")
            
            if docs_sem_limite < count:
                print(f"  ⚠️ PROBLEMA: Count({count}) > Docs retornados({docs_sem_limite})")
                print(f"  💡 SOLUÇÃO: Use collection.get(limit={count}) para pegar todos")
            else:
                print(f"  ✅ Todos os documentos retornados corretamente")
                
        except Exception as e:
            print(f"  ❌ Erro na consulta sem limite: {e}")
        
        return True
        
    except Exception as e:
        print(f"❌ Erro no cliente local: {e}")
        return False

def teste_server_http(collection_name="EC_IFCora-1"):
    """
    Teste via HTTP Client (se servidor estiver rodando)
    """
    print("\n🌐 TESTANDO SERVER HTTP (HttpClient)")
    print("-" * 50)
    
    # Testar várias portas
    portas_teste = [8001, 8000]  # 8001 é a porta correta conforme manage_api.sh
    
    for porta in portas_teste:
        try:
            print(f"🔍 Tentando conectar na porta {porta}...")
            client = chromadb.HttpClient(host="localhost", port=porta)
            
            # Verificar heartbeat
            heartbeat = client.heartbeat()
            print(f"✅ Conectado na porta {porta}! Heartbeat: {heartbeat}")
            
            # Listar coleções
            collections = client.list_collections()
            collection_names = [col.name for col in collections]
            print(f"📚 Coleções disponíveis: {collection_names}")
            
            if not collection_names:
                print("ℹ️  Nenhuma coleção encontrada via HTTP")
                return True
            
            if collection_name in collection_names:
                collection = client.get_collection(name=collection_name)
                count = collection.count()
                print(f"✅ Coleção '{collection_name}': {count} documentos")
                
            return True
            
        except Exception as e:
            print(f"❌ Porta {porta} falhou: {e}")
            continue
    
    print("❌ Nenhuma porta HTTP funcionou - servidor não está rodando")
    return False

def verificar_arquivos_db():
    """
    Verificar se os arquivos do banco existem
    """
    print("\n📁 VERIFICANDO ARQUIVOS DO BANCO")
    print("-" * 40)
    
    db_path = "./chroma_db"
    
    if not os.path.exists(db_path):
        print(f"❌ Diretório {db_path} não existe")
        return False
    
    print(f"✅ Diretório {db_path} existe")
    
    # Listar conteúdo
    try:
        contents = os.listdir(db_path)
        print(f"📄 Conteúdo: {contents}")
        
        # Verificar tamanhos dos arquivos
        for item in contents:
            item_path = os.path.join(db_path, item)
            if os.path.isfile(item_path):
                size = os.path.getsize(item_path)
                print(f"   {item}: {size} bytes")
            else:
                print(f"   {item}: diretório")
                
    except Exception as e:
        print(f"❌ Erro ao listar conteúdo: {e}")
        return False
    
    return True

def main():
    print("🚀 DIAGNÓSTICO CHROMADB - VERSÃO CORRIGIDA")
    print("=" * 55)
    
    # Verificar arquivos
    verificar_arquivos_db()
    
    # Teste 1: Client local (mesmo que a API usa)
    sucesso_local = teste_client_local("EC_IFCora-1")
    
    # Teste 2: Server HTTP (se estiver rodando)
    sucesso_http = teste_server_http("EC_IFCora-1")
    
    # Conclusão
    print(f"\n🎯 RESUMO DIAGNÓSTICO:")
    print("-" * 40)
    
    if sucesso_local:
        print("✅ Cliente local (PersistentClient) funcionando")
    else:
        print("❌ Cliente local com problemas")
    
    if sucesso_http:
        print("✅ Servidor HTTP funcionando")
    else:
        print("⚠️ Servidor HTTP não está rodando")
    
    print(f"\n💡 SOLUÇÕES:")
    print("-" * 40)
    
    if not sucesso_http:
        print("🚀 Para iniciar o servidor:")
        print("   ./manage_api.sh start")
        print("")
    
    if sucesso_local:
        print("✅ O banco de dados local está OK")
        print("   Você pode usar a API mesmo sem servidor HTTP")
    
    print(f"\n🔧 COMANDOS ÚTEIS:")
    print("   ./manage_api.sh status    # Ver status do servidor") 
    print("   ./manage_api.sh start     # Iniciar servidor")
    print("   ./manage_api.sh logs      # Ver logs")

if __name__ == "__main__":
    main()