<?php
/**
 * Arquivo de Configuração - Sistema de Quebra Inteligente de Texto
 * Arquivo: config.php
 * VERSÃO COM SOBREPOSIÇÃO PARA PRESERVAR CONTEXTO
 */

// CONFIGURAÇÕES DA API OPENAI
function getOpenAIConfig() {
    return [
        'api_key' => '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', // DEFINA SUA CHAVE AQUI!
        'api_url' => 'https://api.openai.com/v1/chat/completions',
        'model' => 'gpt-4o',
        'temperature' => 0.1,
        'max_tokens' => 4000
    ];
}

// CONFIGURAÇÕES DE CHUNKING COM SOBREPOSIÇÃO
function getChunkingConfig() {
    return [
        'target_chunk_size' => 10000,       // Tamanho alvo: ~10k caracteres
        'overlap_size' => 600,              // SOBREPOSIÇÃO: ~150 tokens (600 chars)
        'overlap_tokens' => 150,            // Sobreposição em tokens para referência
        'chars_per_token' => 4,             // Aproximação para estimativas
        'min_chunk_size' => 3000,           // Mínimo aceitável
        'max_chunk_size' => 15000,          // Máximo flexível
        'preferred_min' => 8000,            // Mínimo preferencial
        'preferred_max' => 12000,           // Máximo preferencial
        'final_chunk_min' => 1000,          // Mínimo para último chunk
        'use_ai_analysis' => true,          // Usar análise IA
        'fallback_to_rules' => true,        // Fallback se IA falhar
        'semantic_priority' => true,        // SEMPRE priorizar semântica
        'respect_sentence_endings' => true, // OBRIGATÓRIO respeitar pontos finais
        'preserve_concepts' => true,        // Nunca quebrar conceitos
        'flexible_sizing' => true,          // Aceitar variação para semântica
        'force_sentence_breaks' => true,    // FORÇAR quebras apenas em fim de frase
        'overlap_search_range' => 300      // Alcance para buscar início da sobreposição
    ];
}

// PROMPT MELHORADO PARA EVITAR CORTES DE FRASES
function getChunkAnalysisPrompt() {
    return "Você é um especialista em segmentação semântica de textos com PRIORIDADE ABSOLUTA em preservar frases completas.

REGRA FUNDAMENTAL - NUNCA CORTAR FRASES:
JAMAIS faça quebras que deixem frases incompletas. SEMPRE termine chunks em pontos finais (.) seguidos de espaço ou nova linha.

MISSÃO: Identificar pontos onde uma IDEIA/CONCEITO termina COMPLETAMENTE e outro INICIA, criando blocos semanticamente autônomos.

ESTRATÉGIA DE QUEBRAS SEGURAS:
1. OBRIGATÓRIO: Quebrar apenas após pontos finais (. seguido de espaço/nova linha)
2. Identificar onde cada ideia, conceito ou argumento se COMPLETA totalmente
3. Procurar transições naturais entre temas diferentes
4. Buscar conclusões de parágrafos ou seções
5. Respeitar unidades conceituais (problema+solução completa, causa+efeito completo)

PONTOS IDEAIS PARA QUEBRA (sempre após ponto final):
- Fim de uma explicação completa antes de nova explicação
- Conclusão de um argumento antes de argumento diferente
- Término de um exemplo antes de novo exemplo
- Mudança de perspectiva ou abordagem metodológica
- Final de análise antes de nova análise
- Transição entre contexto histórico e aplicação prática

PONTOS ABSOLUTAMENTE PROIBIDOS:
- Meio de frases (sem ponto final antes da quebra)
- Entre sujeito e predicado
- Entre artigo e substantivo
- No meio de citações
- Entre números e suas unidades
- Durante enumerações em andamento
- No meio de explicações causais

VERIFICAÇÃO OBRIGATÓRIA:
Antes de sugerir qualquer quebra, verifique:
1. Há um ponto final (.) imediatamente antes da posição?
2. A frase anterior está completa e faz sentido sozinha?
3. A próxima parte inicia uma nova ideia/conceito?

REFERÊNCIA DE TAMANHO (flexível para preservar frases):
- Preferir chunks entre 8k-12k caracteres
- Aceitar chunks menores (3k-7k) se necessário para não cortar frases
- Aceitar chunks maiores (12k-15k) se a frase/ideia não terminou
- NUNCA sacrificar integridade de frases por tamanho

FORMATO DE RESPOSTA:
QUEBRAS: [posição1, posição2, posição3, ...]
JUSTIFICATIVA: [explicação de cada quebra, confirmando que há ponto final]

Exemplo:
QUEBRAS: [8547, 18923]
JUSTIFICATIVA: [
  'Quebra após ponto final que conclui análise histórica (pos 8547), antes de iniciar metodologia',
  'Quebra após ponto final que finaliza casos práticos (pos 18923), antes das considerações finais'
]

LEMBRE-SE: Sua função é preservar a INTEGRIDADE DAS FRASES acima de tudo. Melhor ter chunks irregulares do que frases cortadas.";
}

// PROMPT PARA ANÁLISE DE QUALIDADE ATUALIZADO
function getQualityAnalysisPrompt() {
    return "Analise a qualidade dos chunks considerando ESPECIALMENTE a integridade das frases:

CRITÉRIOS PRINCIPAIS:
1. INTEGRIDADE DE FRASES: Cada chunk inicia e termina com frases completas?
2. AUTONOMIA SEMÂNTICA: Cada chunk faz sentido independentemente?
3. COESÃO INTERNA: O conteúdo do chunk é coeso e relacionado?
4. COMPLETUDE CONCEITUAL: Ideias e conceitos estão completos?
5. TRANSIÇÕES APROPRIADAS: As quebras foram feitas em pontos semânticos adequados?

VERIFICAÇÕES CRÍTICAS:
- Nenhum chunk pode terminar no meio de uma frase
- Nenhum chunk pode iniciar com pronomes sem referente (ele, ela, isso, aquilo)
- Nenhum chunk pode terminar com vírgulas ou outros indicadores de continuação

FORMATO DE RESPOSTA:
QUALIDADE_GERAL: [EXCELENTE|BOA|REGULAR|INADEQUADA]
PROBLEMAS_FRASES: [problemas de integridade de frases, se houver]
PROBLEMAS_SEMANTICOS: [problemas de coesão semântica, se houver]
RECOMENDAÇÃO: [APROVAR|REPROCESSAR]";
}

// CONFIGURAÇÕES DE VALIDAÇÃO ATUALIZADAS
function getValidationRules() {
    return [
        'min_text_length' => 1000,
        'max_text_length' => 500000,
        'min_chunks' => 2,
        'max_chunks' => 100,
        'quality_threshold' => 0.75,
        'sentence_integrity_required' => true, // Nova validação
        'min_overlap_chars' => 500,            // Mínima sobreposição
        'max_overlap_chars' => 800             // Máxima sobreposição
    ];
}

// CONFIGURAÇÕES DE INTERFACE ATUALIZADAS
function getInterfaceConfig() {
    return [
        'app_name' => 'Sistema de Chunking Semântico',
        'app_subtitle' => 'Segmentação Inteligente com Preservação de Frases',
        'max_processing_time' => 300,
        'show_processing_log' => true,
        'show_statistics' => true,
        'show_chunk_preview' => true,
        'show_overlap_info' => true // Mostrar informações de sobreposição
    ];
}

// REGRAS DE QUEBRA SEMÂNTICA ATUALIZADAS
function getSemanticBreakRules() {
    return [
        // Marcadores de transição forte (só após ponto final)
        'strong_transitions' => [
            'Por outro lado', 'Em contraste', 'Contudo', 'No entanto',
            'Entretanto', 'Todavia', 'Diferentemente', 'Além disso'
        ],
        
        // Introduções de novos temas (só após ponto final)
        'topic_changes' => [
            'Outro aspecto', 'Passando para', 'Vamos abordar',
            'Considerando também', 'Um ponto importante', 'Agora vamos'
        ],
        
        // Conclusões (só após ponto final)
        'conclusions' => [
            'Em resumo', 'Concluindo', 'Para finalizar',
            'Em síntese', 'Portanto', 'Assim', 'Dessa forma'
        ],
        
        // OBRIGATÓRIO: Fim de frase
        'sentence_endings' => ['. ', '.\n', '. "', '."', '.)', '.")', '."]'],
        
        // Padrões que NUNCA devem ser quebrados
        'never_break_patterns' => [
            'no_meio_de_frase' => '/[a-z],?\s*$/',  // Termina sem ponto
            'inicia_pronome' => '/^(ele|ela|isso|isto|eles|elas|aquilo|disso|nisso)\s/i',
            'meio_de_lista' => '/,\s*$/',           // Termina com vírgula
            'meio_de_citacao' => '/"[^"]*$/',       // Aspas não fechadas
        ]
    ];
}

// CONFIGURAÇÕES DE EXPORT ATUALIZADAS
function getExportConfig() {
    return [
        'formats' => [
            'json' => 'JSON estruturado',
            'txt' => 'Texto simples',
            'csv' => 'CSV com metadados',
            'overlap_txt' => 'TXT com sobreposições marcadas'
        ],
        'metadata_fields' => [
            'chunk_id',
            'chunk_size_chars',
            'chunk_size_tokens',
            'chunk_order',
            'quality_score',
            'overlap_start',
            'overlap_end',
            'sentence_integrity'
        ]
    ];
}

// FUNÇÕES DE SOBREPOSIÇÃO
function calculateOverlapPositions($text, $chunkStart, $chunkEnd, $overlapSize) {
    $config = getChunkingConfig();
    $searchRange = $config['overlap_search_range'];
    
    // Calcular início da sobreposição do próximo chunk
    $overlapStart = max(0, $chunkEnd - $overlapSize);
    
    // Buscar início de frase mais próximo para a sobreposição
    $adjustedStart = findSentenceStart($text, $overlapStart, $searchRange);
    
    return [
        'overlap_start' => $adjustedStart,
        'overlap_end' => $chunkEnd,
        'overlap_size' => $chunkEnd - $adjustedStart
    ];
}

// ENCONTRAR INÍCIO DE FRASE PARA SOBREPOSIÇÃO
function findSentenceStart($text, $position, $searchRange = 300) {
    $start = max(0, $position - $searchRange);
    $segment = mb_substr($text, $start, $position - $start + $searchRange);
    
    // Procurar por inícios de frase (após ponto + espaço/nova linha)
    $sentenceStarts = [];
    $patterns = ['. ', '.\n', '. "', '."'];
    
    foreach ($patterns as $pattern) {
        $offset = 0;
        while (($pos = mb_strpos($segment, $pattern, $offset)) !== false) {
            $actualPos = $start + $pos + mb_strlen($pattern);
            if ($actualPos <= $position) {
                $sentenceStarts[] = $actualPos;
            }
            $offset = $pos + 1;
        }
    }
    
    // Se início do texto, usar posição 0
    if (empty($sentenceStarts) && $start === 0) {
        return 0;
    }
    
    // Encontrar início mais próximo da posição desejada
    if (!empty($sentenceStarts)) {
        $closest = $sentenceStarts[0];
        $minDistance = abs($closest - $position);
        
        foreach ($sentenceStarts as $startPos) {
            $distance = abs($startPos - $position);
            if ($distance < $minDistance && $startPos <= $position) {
                $minDistance = $distance;
                $closest = $startPos;
            }
        }
        
        return $closest;
    }
    
    // Fallback: buscar espaço mais próximo
    $spacePos = mb_strrpos(mb_substr($text, $start, $position - $start), ' ');
    return $spacePos !== false ? $start + $spacePos + 1 : $position;
}

// VALIDAÇÃO DE INTEGRIDADE DE FRASES
function validateSentenceIntegrity($chunk, $index) {
    $errors = [];
    $warnings = [];
    
    $text = trim($chunk);
    $rules = getSemanticBreakRules();
    
    // Verificar se termina com ponto final
    if (!preg_match('/[.!?](\s*["\']?)?$/', $text)) {
        $errors[] = "Chunk $index não termina com frase completa";
    }
    
    // Verificar se inicia adequadamente (não com pronome sem contexto)
    if (preg_match($rules['never_break_patterns']['inicia_pronome'], $text)) {
        $errors[] = "Chunk $index inicia com pronome sem referente";
    }
    
    // Verificar se não termina no meio de lista/enumeração
    if (preg_match($rules['never_break_patterns']['meio_de_lista'], $text)) {
        $errors[] = "Chunk $index termina no meio de enumeração";
    }
    
    // Verificar aspas não fechadas
    $quotes = substr_count($text, '"');
    if ($quotes % 2 !== 0) {
        $warnings[] = "Chunk $index pode ter aspas não fechadas";
    }
    
    return [
        'errors' => $errors,
        'warnings' => $warnings,
        'sentence_integrity' => empty($errors) ? 'PRESERVADA' : 'COMPROMETIDA'
    ];
}

// ESTIMATIVA DE TOKENS COM SOBREPOSIÇÃO
function estimateTokenCountWithOverlap($text) {
    $config = getChunkingConfig();
    $baseTokens = ceil(mb_strlen($text) / $config['chars_per_token']);
    
    // Calcular tokens adicionais pela sobreposição
    $estimatedChunks = ceil($baseTokens / ($config['target_chunk_size'] / $config['chars_per_token']));
    $overlapTokens = ($estimatedChunks - 1) * $config['overlap_tokens'];
    
    return $baseTokens + $overlapTokens;
}

// CÁLCULO DE POSIÇÕES COM SOBREPOSIÇÃO
function calculateChunkPositionsWithOverlap($breakPositions, $text) {
    $positions = [];
    $textLength = mb_strlen($text);
    $config = getChunkingConfig();
    $overlapSize = $config['overlap_size'];
    
    for ($i = 0; $i < count($breakPositions); $i++) {
        $start = $i === 0 ? 0 : $breakPositions[$i-1];
        $end = $breakPositions[$i];
        
        // Para chunks após o primeiro, ajustar início para incluir sobreposição
        if ($i > 0) {
            $overlapInfo = calculateOverlapPositions($text, $start, $end, $overlapSize);
            $start = $overlapInfo['overlap_start'];
        }
        
        if ($i === count($breakPositions) - 1) {
            $end = $textLength;
        }
        
        $positions[] = [
            'start' => $start,
            'end' => $end,
            'size' => $end - $start,
            'has_overlap' => $i > 0,
            'overlap_size' => $i > 0 ? $end - $start - ($breakPositions[$i] - $breakPositions[$i-1]) : 0
        ];
    }
    
    return $positions;
}

// VALIDAÇÃO SIMPLES DE QUALIDADE DO CHUNK
function validateChunkQuality($chunk, $index) {
    $errors = [];
    $warnings = [];
    
    $charCount = mb_strlen($chunk);
    $config = getChunkingConfig();
    
    // Verificações flexíveis focadas em semântica
    if ($charCount < $config['final_chunk_min']) {
        $errors[] = "Chunk $index muito pequeno ({$charCount} chars) - pode ter perdido contexto";
    }
    
    if ($charCount > $config['max_chunk_size']) {
        $warnings[] = "Chunk $index muito grande ({$charCount} chars) - verificar se ideia não pode ser dividida";
    }
    
    // Aceitar chunks pequenos se preservam semântica
    if ($charCount >= $config['final_chunk_min'] && $charCount < $config['preferred_min']) {
        $warnings[] = "Chunk $index menor que preferencial ({$charCount} chars) - OK se preserva integridade conceitual";
    }
    
    // Aceitar chunks grandes se preservam semântica  
    if ($charCount > $config['preferred_max'] && $charCount <= $config['max_chunk_size']) {
        $warnings[] = "Chunk $index maior que preferencial ({$charCount} chars) - OK se preserva unidade conceitual";
    }
    
    // Verificação de autonomia semântica
    if (preg_match('/^(ele|ela|isso|isto|eles|elas|tal|aquilo|dessa forma|assim|portanto)\s/i', trim($chunk))) {
        $errors[] = "Chunk $index inicia com referência que depende de contexto anterior";
    }
    
    // Verificação de completude conceitual
    if (preg_match('/\,\s*$/', trim($chunk))) {
        $errors[] = "Chunk $index termina de forma incompleta (ideia cortada)";
    }
    
    // Verificar se chunk não termina no meio de explicação
    if (preg_match('/(isto é|ou seja|por exemplo|como|sendo que)[\s\w]*$/', trim($chunk))) {
        $warnings[] = "Chunk $index pode terminar no meio de explicação";
    }
    
    // Score baseado em preservação semântica, não apenas tamanho
    $semanticScore = 1.0;
    if (!empty($errors)) $semanticScore = 0.3;
    elseif (!empty($warnings)) $semanticScore = 0.7;
    
    // Bonus para chunks que respeitam faixa preferencial E têm boa semântica
    if ($charCount >= $config['preferred_min'] && $charCount <= $config['preferred_max'] && $semanticScore >= 0.7) {
        $semanticScore += 0.2;
    }
    
    return [
        'errors' => $errors,
        'warnings' => $warnings,
        'char_count' => $charCount,
        'token_count' => estimateTokenCount($chunk),
        'quality_score' => min(1.0, $semanticScore),
        'semantic_integrity' => empty($errors) ? 'PRESERVADA' : 'COMPROMETIDA'
    ];
}

// VALIDAÇÃO DE QUALIDADE COM SOBREPOSIÇÃO
function validateChunkQualityWithOverlap($chunk, $index, $hasOverlap = false, $overlapSize = 0) {
    $basicQuality = validateChunkQuality($chunk, $index);
    $sentenceCheck = validateSentenceIntegrity($chunk, $index);
    
    // Combinar validações
    $errors = array_merge($basicQuality['errors'], $sentenceCheck['errors']);
    $warnings = array_merge($basicQuality['warnings'], $sentenceCheck['warnings']);
    
    // Verificar sobreposição adequada
    if ($hasOverlap && $overlapSize < 300) {
        $warnings[] = "Chunk $index tem sobreposição pequena ({$overlapSize} chars)";
    }
    
    // Score considerando integridade de frases
    $semanticScore = $basicQuality['quality_score'];
    if ($sentenceCheck['sentence_integrity'] === 'COMPROMETIDA') {
        $semanticScore *= 0.3; // Penalidade severa por frases cortadas
    }
    
    return [
        'errors' => $errors,
        'warnings' => $warnings,
        'char_count' => $basicQuality['char_count'],
        'token_count' => $basicQuality['token_count'],
        'quality_score' => $semanticScore,
        'sentence_integrity' => $sentenceCheck['sentence_integrity'],
        'has_overlap' => $hasOverlap,
        'overlap_size' => $overlapSize
    ];
}

// VERIFICAÇÃO DE DEPENDÊNCIAS (mantida)
function checkDependencies() {
    $dependencies = [
        'curl' => extension_loaded('curl'),
        'json' => extension_loaded('json'),
        'mbstring' => extension_loaded('mbstring')
    ];
    
    $missing = array_filter($dependencies, function($loaded) {
        return !$loaded;
    });
    
    if (!empty($missing)) {
        throw new Exception('Extensões PHP necessárias: ' . implode(', ', array_keys($missing)));
    }
    
    return true;
}

// VALIDAÇÃO DE CONFIGURAÇÃO (mantida)
function validateConfig() {
    $config = getOpenAIConfig();
    return !empty($config['api_key']) && $config['api_key'] !== 'sua-chave-api-openai-aqui';
}

// ESTIMATIVA DE TOKENS (mantida mas considera sobreposição)
function estimateTokenCount($text) {
    $chunkConfig = getChunkingConfig();
    return ceil(mb_strlen($text) / $chunkConfig['chars_per_token']);
}

// ENCONTRAR MELHOR PONTO DE QUEBRA PRÓXIMO (aprimorada)
function findNearestSentenceEnd($text, $position, $searchRange = 200) {
    $start = max(0, $position - $searchRange);
    $end = min(mb_strlen($text), $position + $searchRange);
    $segment = mb_substr($text, $start, $end - $start);
    
    // Procura por pontos finais com padrões mais específicos
    $sentenceEnds = [];
    $patterns = ['. ', '.\n', '. "', '."', '.)', '.")', '."]'];
    
    foreach ($patterns as $pattern) {
        $offset = 0;
        while (($pos = mb_strpos($segment, $pattern, $offset)) !== false) {
            $sentenceEnds[] = $start + $pos + mb_strlen($pattern);
            $offset = $pos + 1;
        }
    }
    
    if (empty($sentenceEnds)) {
        return $position; // Se não encontrar, mantém posição original
    }
    
    // Encontra o ponto mais próximo da posição desejada
    $closest = $sentenceEnds[0];
    $minDistance = abs($closest - $position);
    
    foreach ($sentenceEnds as $end) {
        $distance = abs($end - $position);
        if ($distance < $minDistance) {
            $minDistance = $distance;
            $closest = $end;
        }
    }
    
    return $closest;
}
?>