<?php
/**
 * Processador Principal - Sistema de Chunking com Sobreposição
 * Arquivo: processor.php
 * VERSÃO COM SOBREPOSIÇÃO DE 150 TOKENS PARA PRESERVAR CONTEXTO
 */

require_once 'config.php';

class TextChunker {
    
    private $apiKey;
    private $apiUrl;
    private $model;
    private $config;
    private $progressCallback;
    
    public function __construct() {
        $apiConfig = getOpenAIConfig();
        $this->apiKey = $apiConfig['api_key'];
        $this->apiUrl = $apiConfig['api_url'];
        $this->model = $apiConfig['model'];
        $this->config = getChunkingConfig();
        $this->progressCallback = null;
    }
    
    /**
     * Processa o texto dividindo em chunks semânticos com sobreposição
     */
    public function processText($text, $progressCallback = null) {
        try {
            $this->progressCallback = $progressCallback;
            
            $this->log("Iniciando análise com sobreposição de {$this->config['overlap_tokens']} tokens...");
            
            // Análise básica do texto
            $textStats = $this->analyzeText($text);
            
            $this->log("Texto: {$textStats['chars']} caracteres, {$textStats['sentences']} frases");
            
            // Decidir método de chunking
            $useAI = $this->config['use_ai_analysis'] && validateConfig();
            
            if ($useAI) {
                $this->log("Usando análise IA para quebras semânticas seguras...");
                $breakPositions = $this->analyzeWithAI($text);
                
                if (!$breakPositions && $this->config['fallback_to_rules']) {
                    $this->log("Fallback: Usando regras estruturais com preservação de frases...");
                    $breakPositions = $this->analyzeWithRules($text);
                }
            } else {
                $this->log("Usando regras estruturais com preservação de frases...");
                $breakPositions = $this->analyzeWithRules($text);
            }
            
            if (!$breakPositions) {
                throw new Exception("Falha na identificação de pontos de quebra seguros");
            }
            
            $this->log("Gerando " . count($breakPositions) . " chunks com sobreposição...");
            
            // Gerar chunks com sobreposição
            $chunks = $this->generateChunksWithOverlap($text, $breakPositions);
            
            $this->log("Validando integridade de frases e qualidade semântica...");
            
            // Análise de qualidade com foco em integridade
            $qualityAnalysis = $this->analyzeQualityWithOverlap($chunks);
            
            $this->log("Processamento concluído! Qualidade: " . $qualityAnalysis['grade']);
            $this->log("Sobreposição média: " . round($qualityAnalysis['avg_overlap_size']) . " caracteres");
            
            return [
                'chunks' => $chunks,
                'statistics' => $textStats,
                'quality' => $qualityAnalysis,
                'break_positions' => $breakPositions,
                'metadata' => $this->generateMetadata($text, $chunks, $qualityAnalysis),
                'overlap_info' => $this->generateOverlapInfo($chunks)
            ];
            
        } catch (Exception $e) {
            $this->log("ERRO: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Análise básica do texto
     */
    private function analyzeText($text) {
        $sentences = $this->countSentences($text);
        $estimatedTokens = estimateTokenCountWithOverlap($text);
        
        return [
            'chars' => mb_strlen($text),
            'words' => str_word_count($text),
            'sentences' => $sentences,
            'paragraphs' => count(preg_split('/\n\s*\n/', $text, -1, PREG_SPLIT_NO_EMPTY)),
            'estimated_tokens' => $estimatedTokens,
            'estimated_tokens_with_overlap' => $estimatedTokens
        ];
    }
    
    /**
     * Contar frases de forma mais precisa
     */
    private function countSentences($text) {
        $patterns = ['. ', '.\n', '! ', '!\n', '? ', '?\n', '. "', '."', '! "', '!"', '? "', '?"'];
        $count = 0;
        
        foreach ($patterns as $pattern) {
            $count += substr_count($text, $pattern);
        }
        
        // Verificar se termina com ponto final
        if (preg_match('/[.!?](\s*["\']?)?$/', trim($text))) {
            $count++;
        }
        
        return max(1, $count);
    }
    
    /**
     * Análise com IA focada em preservação de frases
     */
    private function analyzeWithAI($text) {
        try {
            $prompt = getChunkAnalysisPrompt();
            
            // Adicionar informação sobre sobreposição no prompt
            $contextPrompt = $prompt . "\n\nIMPORTANTE: Os chunks terão sobreposição de {$this->config['overlap_tokens']} tokens (~{$this->config['overlap_size']} caracteres) para preservar contexto. Isso será adicionado automaticamente, então foque apenas em identificar pontos de quebra semântica APÓS pontos finais.";
            
            $data = [
                'model' => $this->model,
                'messages' => [
                    ['role' => 'system', 'content' => $contextPrompt],
                    ['role' => 'user', 'content' => $text]
                ],
                'temperature' => 0.1,
                'max_tokens' => 4000
            ];
            
            $response = $this->callOpenAI($data);
            
            // Extrair posições da resposta
            if (preg_match('/QUEBRAS:\s*\[([\d,\s]+)\]/', $response, $matches)) {
                $positions = array_map('intval', array_filter(array_map('trim', explode(',', $matches[1]))));
                
                // Extrair justificativas
                $justifications = [];
                if (preg_match('/JUSTIFICATIVA:\s*\[(.*?)\]/s', $response, $justMatches)) {
                    $justText = $justMatches[1];
                    preg_match_all("/'([^']+)'/", $justText, $justArray);
                    $justifications = $justArray[1];
                }
                
                $this->log("IA identificou " . count($positions) . " pontos de quebra seguros");
                foreach ($justifications as $i => $just) {
                    $this->log("Quebra " . ($i+1) . ": " . $just);
                }
                
                // Validar que todas as quebras estão em fins de frase
                $validatedPositions = $this->validateSentenceBreaks($text, $positions);
                
                return $validatedPositions;
            }
            
            $this->log("IA não retornou formato válido de quebras");
            return false;
            
        } catch (Exception $e) {
            $this->log("Erro na análise IA: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Validar que quebras estão em fins de frase
     */
    private function validateSentenceBreaks($text, $positions) {
        $validated = [];
        $textLength = mb_strlen($text);
        
        foreach ($positions as $pos) {
            if ($pos > 0 && $pos <= $textLength) {
                // Verificar se há ponto final antes da posição
                $beforeChar = mb_substr($text, $pos - 1, 1);
                $beforeTwoChars = mb_substr($text, $pos - 2, 2);
                
                $isSafeBreak = false;
                
                // Verificar padrões de fim de frase
                $safePatterns = ['. ', '.\n', '. "', '."', '.)', '.")', '."]'];
                foreach ($safePatterns as $pattern) {
                    if (mb_substr($text, $pos - mb_strlen($pattern), mb_strlen($pattern)) === $pattern) {
                        $isSafeBreak = true;
                        break;
                    }
                }
                
                if ($isSafeBreak) {
                    $validated[] = $pos;
                } else {
                    // Ajustar para fim de frase mais próximo
                    $adjustedPos = findNearestSentenceEnd($text, $pos);
                    if ($adjustedPos !== $pos && $adjustedPos <= $textLength) {
                        $this->log("Ajustando quebra de posição $pos para $adjustedPos (fim de frase)");
                        $validated[] = $adjustedPos;
                    }
                }
            }
        }
        
        // Garantir final do texto
        if (empty($validated) || end($validated) !== $textLength) {
            $validated[] = $textLength;
        }
        
        sort($validated);
        return array_unique($validated);
    }
    
    /**
     * Análise com regras estruturais priorizando fins de frase
     */
    private function analyzeWithRules($text) {
        $breakPositions = [];
        $textLength = mb_strlen($text);
        $currentPos = 0;
        
        while ($currentPos < $textLength) {
            $nextBreak = $this->findNextSafeBreak($text, $currentPos);
            
            if ($nextBreak > $currentPos && $nextBreak < $textLength) {
                $breakPositions[] = $nextBreak;
                $currentPos = $nextBreak;
            } else {
                break;
            }
        }
        
        // Garantir final do texto
        if (empty($breakPositions) || end($breakPositions) !== $textLength) {
            $breakPositions[] = $textLength;
        }
        
        return $breakPositions;
    }
    
    /**
     * Encontra próxima quebra segura (sempre em fim de frase)
     */
    private function findNextSafeBreak($text, $start) {
        $targetSize = $this->config['target_chunk_size'];
        $maxSize = $this->config['max_chunk_size'];
        $minSize = $this->config['min_chunk_size'];
        
        $textLength = mb_strlen($text);
        $remainingText = $textLength - $start;
        
        // Se texto restante é pequeno, usar tudo
        if ($remainingText <= $maxSize) {
            return $textLength;
        }
        
        // Encontrar ponto ideal
        $idealEnd = $start + $targetSize;
        
        // Buscar quebra segura em uma janela ao redor do ponto ideal
        $searchStart = max($start + $minSize, $idealEnd - 1000);
        $searchEnd = min($idealEnd + 1000, $start + $maxSize);
        
        $safeBreak = $this->findSafeBreakPoint($text, $searchStart, $searchEnd);
        
        if ($safeBreak !== false) {
            return $safeBreak;
        }
        
        // Se não encontrou quebra segura, procurar mais adiante
        $extendedEnd = min($start + $maxSize, $textLength);
        $safeBreak = $this->findSafeBreakPoint($text, $searchStart, $extendedEnd);
        
        return $safeBreak !== false ? $safeBreak : min($start + $maxSize, $textLength);
    }
    
    /**
     * Encontra ponto de quebra seguro (SEMPRE em fim de frase)
     */
    private function findSafeBreakPoint($text, $start, $end) {
        $searchText = mb_substr($text, $start, $end - $start);
        
        // 1. Procurar fim de frases com marcadores semânticos
        $semanticBreak = $this->findSemanticSentenceEnds($searchText, $start);
        if ($semanticBreak !== false) return $semanticBreak;
        
        // 2. Procurar fim de parágrafos
        $paragraphBreak = $this->findParagraphSentenceEnd($searchText, $start);
        if ($paragraphBreak !== false) return $paragraphBreak;
        
        // 3. Procurar qualquer fim de frase
        $sentenceBreak = $this->findAnySentenceEnd($searchText, $start);
        if ($sentenceBreak !== false) return $sentenceBreak;
        
        return false; // Nenhuma quebra segura encontrada
    }
    
    /**
     * Procurar fins de frase com marcadores semânticos
     */
    private function findSemanticSentenceEnds($text, $offset) {
        $rules = getSemanticBreakRules();
        $sentenceEnds = ['. ', '.\n', '. "', '."'];
        
        // Procurar marcadores semânticos seguidos de ponto final
        $allMarkers = array_merge(
            $rules['strong_transitions'],
            $rules['topic_changes'],
            $rules['conclusions']
        );
        
        foreach ($allMarkers as $marker) {
            foreach ($sentenceEnds as $ending) {
                $pattern = $marker . $ending;
                $pos = mb_strripos($text, $pattern);
                if ($pos !== false) {
                    return $offset + $pos + mb_strlen($pattern);
                }
            }
        }
        
        return false;
    }
    
    /**
     * Procurar fim de parágrafo que termine com frase completa
     */
    private function findParagraphSentenceEnd($text, $offset) {
        $sentenceEnds = ['. ', '.\n', '. "', '."'];
        
        foreach ($sentenceEnds as $ending) {
            $pos = mb_strripos($text, $ending . "\n");
            if ($pos !== false) {
                return $offset + $pos + mb_strlen($ending);
            }
        }
        
        return false;
    }
    
    /**
     * Procurar qualquer fim de frase
     */
    private function findAnySentenceEnd($text, $offset) {
        $sentenceEnds = ['. ', '.\n', '. "', '."', '.)', '.")', '."]'];
        $bestPos = false;
        
        foreach ($sentenceEnds as $ending) {
            $pos = mb_strripos($text, $ending);
            if ($pos !== false) {
                $actualPos = $offset + $pos + mb_strlen($ending);
                if ($bestPos === false || $actualPos > $bestPos) {
                    $bestPos = $actualPos;
                }
            }
        }
        
        return $bestPos;
    }
    
    /**
     * Gera chunks com sobreposição preservando integridade de frases
     */
    private function generateChunksWithOverlap($text, $breakPositions) {
        $chunks = [];
        $textLength = mb_strlen($text);
        $overlapSize = $this->config['overlap_size'];
        
        for ($i = 0; $i < count($breakPositions); $i++) {
            $chunkEnd = $breakPositions[$i];
            $chunkStart = 0;
            $hasOverlap = false;
            $overlapStart = 0;
            
            if ($i === 0) {
                // Primeiro chunk: do início até primeira quebra
                $chunkStart = 0;
            } else {
                // Chunks subsequentes: incluir sobreposição
                $previousEnd = $breakPositions[$i-1];
                $overlapStart = max(0, $previousEnd - $overlapSize);
                
                // Ajustar início da sobreposição para início de frase
                $chunkStart = findSentenceStart($text, $overlapStart);
                $hasOverlap = true;
            }
            
            // Último chunk vai até o final
            if ($i === count($breakPositions) - 1) {
                $chunkEnd = $textLength;
            }
            
            $chunkText = mb_substr($text, $chunkStart, $chunkEnd - $chunkStart);
            $chunkText = trim($chunkText);
            
            if (!empty($chunkText)) {
                $actualOverlapSize = $hasOverlap ? ($breakPositions[$i-1] - $chunkStart) : 0;
                
                $chunks[] = [
                    'id' => $i + 1,
                    'text' => $chunkText,
                    'start_pos' => $chunkStart,
                    'end_pos' => $chunkEnd,
                    'char_count' => mb_strlen($chunkText),
                    'token_estimate' => estimateTokenCount($chunkText),
                    'has_overlap' => $hasOverlap,
                    'overlap_size' => $actualOverlapSize,
                    'overlap_start' => $hasOverlap ? $chunkStart : null,
                    'overlap_end' => $hasOverlap ? $breakPositions[$i-1] : null
                ];
            }
        }
        
        return $chunks;
    }
    
    /**
     * Análise de qualidade com foco em sobreposição e integridade
     */
    private function analyzeQualityWithOverlap($chunks) {
        $totalChunks = count($chunks);
        $qualityScores = [];
        $sentenceIntegrityScores = [];
        $overlapSizes = [];
        $problems = [];
        
        foreach ($chunks as $chunk) {
            // Validação de qualidade básica + integridade de frases
            $quality = validateChunkQualityWithOverlap(
                $chunk['text'], 
                $chunk['id'], 
                $chunk['has_overlap'], 
                $chunk['overlap_size']
            );
            
            $qualityScores[] = $quality['quality_score'];
            $sentenceIntegrityScores[] = $quality['sentence_integrity'] === 'PRESERVADA' ? 1.0 : 0.0;
            
            if ($chunk['has_overlap']) {
                $overlapSizes[] = $chunk['overlap_size'];
            }
            
            if (!empty($quality['errors'])) {
                $problems = array_merge($problems, $quality['errors']);
            }
        }
        
        $avgQuality = array_sum($qualityScores) / count($qualityScores);
        $sentenceIntegrityRate = array_sum($sentenceIntegrityScores) / count($sentenceIntegrityScores);
        $avgOverlapSize = !empty($overlapSizes) ? array_sum($overlapSizes) / count($overlapSizes) : 0;
        
        // Penalizar se muitas frases foram cortadas
        if ($sentenceIntegrityRate < 0.9) {
            $avgQuality *= 0.5; // Penalidade severa
        }
        
        // Classificação considerando integridade de frases
        $grade = 'INADEQUADA';
        if ($avgQuality >= 0.9 && $sentenceIntegrityRate >= 0.95) {
            $grade = 'EXCELENTE';
        } elseif ($avgQuality >= 0.8 && $sentenceIntegrityRate >= 0.9) {
            $grade = 'BOA';
        } elseif ($avgQuality >= 0.7 && $sentenceIntegrityRate >= 0.8) {
            $grade = 'REGULAR';
        }
        
        return [
            'grade' => $grade,
            'average_score' => $avgQuality,
            'sentence_integrity_rate' => $sentenceIntegrityRate,
            'chunk_scores' => $qualityScores,
            'total_chunks' => $totalChunks,
            'problems' => $problems,
            'avg_overlap_size' => $avgOverlapSize,
            'overlap_token_estimate' => round($avgOverlapSize / $this->config['chars_per_token']),
            'recommendation' => (empty($problems) && $sentenceIntegrityRate >= 0.9) ? 'APROVAR' : 'REPROCESSAR',
            'chunks_with_overlap' => count(array_filter($chunks, function($c) { return $c['has_overlap']; }))
        ];
    }
    
    /**
     * Gera informações detalhadas sobre sobreposição
     */
    private function generateOverlapInfo($chunks) {
        $overlapInfo = [];
        
        foreach ($chunks as $chunk) {
            if ($chunk['has_overlap']) {
                $overlapText = mb_substr($chunk['text'], 0, $chunk['overlap_size']);
                $overlapInfo[] = [
                    'chunk_id' => $chunk['id'],
                    'overlap_size_chars' => $chunk['overlap_size'],
                    'overlap_size_tokens' => round($chunk['overlap_size'] / $this->config['chars_per_token']),
                    'overlap_text_preview' => mb_substr($overlapText, 0, 100) . (mb_strlen($overlapText) > 100 ? '...' : ''),
                    'overlap_start_pos' => $chunk['overlap_start'],
                    'overlap_end_pos' => $chunk['overlap_end']
                ];
            }
        }
        
        return $overlapInfo;
    }
    
    /**
     * Gera metadados com informações de sobreposição
     */
    private function generateMetadata($originalText, $chunks, $quality) {
        $totalOverlapChars = array_sum(array_column(array_filter($chunks, function($c) { return $c['has_overlap']; }), 'overlap_size'));
        
        return [
            'source_hash' => md5($originalText),
            'generated_at' => date('Y-m-d H:i:s'),
            'chunking_method' => $this->config['use_ai_analysis'] ? 'AI_ANALYSIS_WITH_OVERLAP' : 'STRUCTURAL_RULES_WITH_OVERLAP',
            'overlap_enabled' => true,
            'overlap_size_chars' => $this->config['overlap_size'],
            'overlap_size_tokens' => $this->config['overlap_tokens'],
            'total_chunks' => count($chunks),
            'chunks_with_overlap' => count(array_filter($chunks, function($c) { return $c['has_overlap']; })),
            'total_overlap_chars' => $totalOverlapChars,
            'total_chars' => mb_strlen($originalText),
            'total_chars_with_overlap' => mb_strlen($originalText) + $totalOverlapChars,
            'total_tokens_estimate' => estimateTokenCount($originalText),
            'total_tokens_with_overlap' => estimateTokenCountWithOverlap($originalText),
            'quality_grade' => $quality['grade'],
            'sentence_integrity_rate' => $quality['sentence_integrity_rate'],
            'average_chunk_size' => round(mb_strlen($originalText) / count($chunks)),
            'config_used' => $this->config
        ];
    }
    
    /**
     * Chamada para OpenAI (mantida)
     */
    private function callOpenAI($data) {
        $headers = [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $this->apiKey
        ];
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $this->apiUrl,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($data),
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 120,
            CURLOPT_SSL_VERIFYPEER => true
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        
        if (curl_error($ch)) {
            curl_close($ch);
            throw new Exception('Erro de conexão: ' . curl_error($ch));
        }
        
        curl_close($ch);
        
        if ($httpCode !== 200) {
            $errorInfo = json_decode($response, true);
            $errorMsg = isset($errorInfo['error']['message']) ? $errorInfo['error']['message'] : 'Erro HTTP: ' . $httpCode;
            throw new Exception($errorMsg);
        }
        
        $result = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('Erro ao processar resposta da API');
        }
        
        if (!isset($result['choices'][0]['message']['content'])) {
            throw new Exception('Resposta inválida da API');
        }
        
        return trim($result['choices'][0]['message']['content']);
    }
    
    /**
     * Log de progresso
     */
    private function log($message) {
        if ($this->progressCallback) {
            call_user_func($this->progressCallback, $message);
        }
    }
    
    /**
     * Export dos chunks com informações de sobreposição
     */
    public function exportChunks($result, $format = 'json') {
        switch ($format) {
            case 'json':
                return json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
            
            case 'csv':
                $csv = "id,text,char_count,token_estimate,has_overlap,overlap_size,start_pos,end_pos\n";
                foreach ($result['chunks'] as $chunk) {
                    $csv .= sprintf('"%d","%s","%d","%d","%s","%d","%d","%d"' . "\n",
                        $chunk['id'],
                        str_replace('"', '""', $chunk['text']),
                        $chunk['char_count'],
                        $chunk['token_estimate'],
                        $chunk['has_overlap'] ? 'sim' : 'não',
                        $chunk['overlap_size'],
                        $chunk['start_pos'],
                        $chunk['end_pos']
                    );
                }
                return $csv;
            
            case 'txt':
                $txt = "=== CHUNKS COM SOBREPOSIÇÃO ===\n\n";
                foreach ($result['chunks'] as $chunk) {
                    $txt .= "CHUNK {$chunk['id']}\n";
                    $txt .= "Tamanho: {$chunk['char_count']} chars (~{$chunk['token_estimate']} tokens)\n";
                    if ($chunk['has_overlap']) {
                        $txt .= "Sobreposição: {$chunk['overlap_size']} chars\n";
                    }
                    $txt .= str_repeat("-", 50) . "\n";
                    $txt .= $chunk['text'] . "\n\n";
                }
                return $txt;
            
            case 'overlap_txt':
                $txt = "=== CHUNKS COM MARCAÇÃO DE SOBREPOSIÇÃO ===\n\n";
                foreach ($result['chunks'] as $chunk) {
                    $txt .= "CHUNK {$chunk['id']}\n";
                    if ($chunk['has_overlap']) {
                        $overlapText = mb_substr($chunk['text'], 0, $chunk['overlap_size']);
                        $newText = mb_substr($chunk['text'], $chunk['overlap_size']);
                        
                        $txt .= "[SOBREPOSIÇÃO: {$chunk['overlap_size']} chars]\n";
                        $txt .= $overlapText . "\n";
                        $txt .= "[NOVO CONTEÚDO]\n";
                        $txt .= $newText . "\n\n";
                    } else {
                        $txt .= "[SEM SOBREPOSIÇÃO]\n";
                        $txt .= $chunk['text'] . "\n\n";
                    }
                    $txt .= str_repeat("=", 50) . "\n\n";
                }
                return $txt;
            
            default:
                throw new Exception("Formato não suportado: $format");
        }
    }
}

// PROCESSAMENTO DO FORMULÁRIO (atualizado)
$result = null;
$error = '';
$processingLog = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!validateConfig()) {
        $error = 'Configure sua chave da API OpenAI no arquivo config.php.';
    } else {
        $inputText = trim($_POST['input_text'] ?? '');
        $exportFormat = $_POST['export_format'] ?? 'json';
        $validationRules = getValidationRules();
        
        if (empty($inputText)) {
            $error = 'Texto obrigatório.';
        } elseif (strlen($inputText) < $validationRules['min_text_length']) {
            $error = 'Texto muito curto. Mínimo: ' . number_format($validationRules['min_text_length']) . ' caracteres.';
        } elseif (strlen($inputText) > $validationRules['max_text_length']) {
            $error = 'Texto muito longo. Máximo: ' . number_format($validationRules['max_text_length']) . ' caracteres.';
        } else {
            try {
                $logCallback = function($message) use (&$processingLog) {
                    $processingLog[] = date('H:i:s') . ' - ' . $message;
                };
                
                $chunker = new TextChunker();
                $result = $chunker->processText($inputText, $logCallback);
                
                if (!$result) {
                    $error = 'Falha no processamento do texto.';
                }
                
            } catch (Exception $e) {
                $error = 'Erro: ' . $e->getMessage();
            }
        }
    }
}

$pageTitle = "Sistema de Chunking Semântico com Sobreposição";
$pageData = compact('result', 'error', 'processingLog');

include 'template.php';
?>