<?php
/**
 * Template HTML Simplificado - Interface Web para Chunking Semântico
 * Arquivo: template.php
 * VERSÃO SIMPLIFICADA COM EXIBIÇÃO DIRETA DOS CHUNKS
 */

// Configurações básicas (evitar dependências de funções externas)
$interfaceConfig = [
    'app_name' => 'Sistema de Chunking Semântico',
    'app_subtitle' => 'Segmentação Inteligente com Preservação de Frases',
    'show_processing_log' => true,
    'show_statistics' => true
];

$validationRules = [
    'min_text_length' => 1000,
    'max_text_length' => 500000
];

$chunkConfig = [
    'target_chunk_size' => 10000,
    'overlap_tokens' => 150,
    'chars_per_token' => 4,
    'semantic_priority' => true,
    'use_ai_analysis' => true,
    'force_sentence_breaks' => true
];

$exportConfig = [
    'formats' => [
        'json' => 'JSON estruturado',
        'txt' => 'Texto simples',
        'csv' => 'CSV com metadados'
    ]
];

// Função auxiliar para acessar dados com fallback
function getDataSafely($array, $key, $default = null) {
    return isset($array[$key]) ? $array[$key] : $default;
}

// Função para verificar se API está configurada
function checkApiConfig() {
    // Verificação básica - substitua pela chave real se necessário
    return true; // Por ora, assumir que está configurado
}
function getQualityCssClass($quality) {
    if (!$quality) return 'unknown';
    
    $grade = is_array($quality) ? 
        getDataSafely($quality, 'grade', getDataSafely($quality, 'quality', 'UNKNOWN')) : 
        $quality;
    
    switch (strtoupper($grade)) {
        case 'EXCELENTE': return 'excellent';
        case 'BOA': return 'good';
        case 'REGULAR': return 'regular';
        case 'INADEQUADA': return 'poor';
        default: return 'unknown';
    }
}

// Função para gerar texto simples com todos os chunks incluindo sobreposição
function generateSimpleChunksText($chunks) {
    $output = "";
    foreach ($chunks as $index => $chunk) {
        if ($index > 0) {
            $output .= "\n" . str_repeat("-", 50) . "\n\n";
        }
        
        $overlapInfo = "";
        if (getDataSafely($chunk, 'has_overlap', false)) {
            $overlapSize = getDataSafely($chunk, 'overlap_size', 0);
            $overlapTokens = round($overlapSize / 4); // estimativa de tokens
            $overlapInfo = " | Sobreposição: {$overlapSize} chars (~{$overlapTokens} tokens)";
        }
        
        $output .= "CHUNK " . ($index + 1) . " (" . number_format(getDataSafely($chunk, 'char_count', 0)) . " chars{$overlapInfo})\n\n";
        $output .= getDataSafely($chunk, 'text', '') . "\n";
    }
    return $output;
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($pageTitle ?? $interfaceConfig['app_name']) ?></title>
    
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
            line-height: 1.6;
            color: #1a202c;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .header {
            text-align: center;
            margin-bottom: 30px;
            color: white;
        }
        
        .header h1 {
            font-size: 2.5em;
            margin-bottom: 10px;
            text-shadow: 0 2px 4px rgba(0,0,0,0.3);
        }
        
        .header p {
            font-size: 1.2em;
            opacity: 0.9;
        }
        
        .content {
            background: white;
            border-radius: 12px;
            padding: 30px;
            box-shadow: 0 10px 25px rgba(0,0,0,0.15);
        }
        
        .error, .info, .success {
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 20px;
            border-left: 4px solid;
        }
        
        .error {
            background: #fef2f2;
            border-color: #dc2626;
            color: #991b1b;
        }
        
        .info {
            background: #eff6ff;
            border-color: #2563eb;
            color: #1e40af;
        }
        
        .success {
            background: #f0fdf4;
            border-color: #16a34a;
            color: #15803d;
        }
        
        .config-info {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
            margin-bottom: 25px;
            padding: 20px;
            background: #f8fafc;
            border-radius: 8px;
        }
        
        .config-item {
            text-align: center;
        }
        
        .config-value {
            font-size: 1.5em;
            font-weight: bold;
            color: #4f46e5;
        }
        
        .config-label {
            font-size: 0.9em;
            color: #64748b;
            margin-top: 5px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #374151;
        }
        
        .form-group small {
            display: block;
            margin-top: 5px;
            color: #6b7280;
        }
        
        textarea, select {
            width: 100%;
            padding: 12px;
            border: 2px solid #e5e7eb;
            border-radius: 8px;
            font-size: 14px;
            transition: border-color 0.3s;
            resize: vertical;
            min-height: 250px;
        }
        
        textarea:focus, select:focus {
            outline: none;
            border-color: #4f46e5;
        }
        
        .char-count {
            margin-top: 8px;
            font-size: 0.9em;
            color: #6b7280;
        }
        
        .char-count .warning {
            color: #d97706;
        }
        
        .char-count .error {
            color: #dc2626;
        }
        
        .form-row {
            display: flex;
            gap: 20px;
            align-items: end;
            flex-wrap: wrap;
        }
        
        .form-actions {
            flex: 1;
        }
        
        .btn {
            background: #4f46e5;
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 8px;
            cursor: pointer;
            font-size: 14px;
            font-weight: 600;
            transition: all 0.3s;
            margin-right: 10px;
        }
        
        .btn:hover:not(:disabled) {
            background: #3730a3;
            transform: translateY(-1px);
        }
        
        .btn:disabled {
            background: #9ca3af;
            cursor: not-allowed;
            transform: none;
        }
        
        .btn-secondary {
            background: #6b7280;
        }
        
        .btn-secondary:hover {
            background: #4b5563;
        }
        
        .btn-export {
            background: #059669;
            font-size: 12px;
            padding: 8px 16px;
            margin: 2px;
        }
        
        .btn-export:hover {
            background: #047857;
        }
        
        .loading {
            display: none;
            text-align: center;
            padding: 40px;
            background: #f8fafc;
            border-radius: 8px;
            margin: 20px 0;
        }
        
        .spinner {
            width: 40px;
            height: 40px;
            border: 4px solid #e5e7eb;
            border-top: 4px solid #4f46e5;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin: 0 auto 15px;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .log {
            background: #1f2937;
            color: #f9fafb;
            padding: 20px;
            border-radius: 8px;
            margin: 20px 0;
            font-family: 'Courier New', monospace;
            max-height: 300px;
            overflow-y: auto;
        }
        
        .log h4 {
            color: #60a5fa;
            margin-bottom: 15px;
        }
        
        .log-item {
            margin-bottom: 5px;
            padding: 2px 0;
        }
        
        .stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
            gap: 15px;
            margin: 20px 0;
            padding: 20px;
            background: #f8fafc;
            border-radius: 8px;
        }
        
        .stat-item {
            text-align: center;
        }
        
        .stat-number {
            font-size: 1.8em;
            font-weight: bold;
            color: #059669;
        }
        
        .stat-label {
            font-size: 0.9em;
            color: #64748b;
            margin-top: 5px;
        }
        
        .quality-indicator {
            padding: 4px 12px;
            border-radius: 20px;
            font-weight: bold;
            font-size: 0.9em;
        }
        
        .quality-excellent {
            background: #dcfce7;
            color: #166534;
        }
        
        .quality-good {
            background: #dbeafe;
            color: #1e40af;
        }
        
        .quality-regular {
            background: #fef3c7;
            color: #92400e;
        }
        
        .quality-poor {
            background: #fecaca;
            color: #991b1b;
        }
        
        .quality-unknown {
            background: #f3f4f6;
            color: #6b7280;
        }
        
        .chunks-result {
            margin-top: 25px;
        }
        
        .result-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            flex-wrap: wrap;
            gap: 15px;
        }
        
        .export-controls {
            display: flex;
            gap: 5px;
            flex-wrap: wrap;
        }
        
        .chunks-output {
            position: relative;
        }
        
        .chunks-textarea {
            width: 100%;
            min-height: 400px;
            font-family: 'Courier New', monospace;
            font-size: 13px;
            line-height: 1.4;
            padding: 15px;
            border: 2px solid #e5e7eb;
            border-radius: 8px;
            background: #fafafa;
            resize: vertical;
        }
        
        .copy-button {
            position: absolute;
            top: 10px;
            right: 10px;
            background: #4f46e5;
            color: white;
            border: none;
            padding: 8px 12px;
            border-radius: 6px;
            cursor: pointer;
            font-size: 12px;
            z-index: 1;
        }
        
        .copy-button:hover {
            background: #3730a3;
        }
        
        .copy-button.copied {
            background: #16a34a;
        }
        
        @media (max-width: 768px) {
            .container {
                padding: 10px;
            }
            
            .content {
                padding: 20px;
            }
            
            .form-row {
                flex-direction: column;
            }
            
            .result-header {
                flex-direction: column;
                align-items: stretch;
            }
            
            .export-controls {
                justify-content: center;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1><?= htmlspecialchars($interfaceConfig['app_name']) ?></h1>
            <p><?= htmlspecialchars($interfaceConfig['app_subtitle']) ?></p>
        </div>
        
        <div class="content">
            <?php if (!empty($pageData['error'])): ?>
                <div class="error">
                    <strong>Erro:</strong> <?= htmlspecialchars($pageData['error']) ?>
                </div>
            <?php endif; ?>
            
            <?php if (!checkApiConfig()): ?>
                <div class="info">
                    <strong>Configuração:</strong> 
                    Para usar análise semântica com IA, configure sua chave da API OpenAI no arquivo <code>config.php</code>.
                    Sem a chave, o sistema usará apenas regras semânticas automáticas.
                </div>
            <?php endif; ?>
            
            <div class="config-info">
                <div class="config-item">
                    <div class="config-value"><?= number_format($chunkConfig['target_chunk_size']) ?></div>
                    <div class="config-label">Tamanho Alvo</div>
                </div>
                <div class="config-item">
                    <div class="config-value"><?= $chunkConfig['overlap_tokens'] ?> tokens</div>
                    <div class="config-label">Sobreposição</div>
                </div>
                <div class="config-item">
                    <div class="config-value"><?= $chunkConfig['semantic_priority'] ? 'Sim' : 'Não' ?></div>
                    <div class="config-label">Prioridade Semântica</div>
                </div>
                <div class="config-item">
                    <div class="config-value"><?= $chunkConfig['use_ai_analysis'] ? 'Ativa' : 'Inativa' ?></div>
                    <div class="config-label">Análise IA</div>
                </div>
                <div class="config-item">
                    <div class="config-value"><?= $chunkConfig['force_sentence_breaks'] ? 'Sim' : 'Não' ?></div>
                    <div class="config-label">Preservar Frases</div>
                </div>
            </div>
            
            <form method="POST" id="chunkingForm">
                <div class="form-group">
                    <label for="input_text">
                        Texto para Chunking Semântico <span style="color: red;">*</span>
                        <small>Cole o texto que será dividido em chunks preservando o contexto semântico</small>
                    </label>
                    <textarea 
                        id="input_text" 
                        name="input_text" 
                        placeholder="Cole aqui o texto para ser dividido em chunks semânticos..."
                        required
                    ><?= htmlspecialchars($_POST['input_text'] ?? '') ?></textarea>
                    <div class="char-count">
                        <span id="charCount">0</span> caracteres • 
                        <span id="tokenEstimate">0</span> tokens estimados • 
                        <span id="chunkEstimate">0</span> chunks estimados
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-actions">
                        <button type="submit" class="btn" id="submitBtn">
                            Processar Chunks
                        </button>
                        <button type="button" class="btn btn-secondary" onclick="clearForm()">
                            Limpar
                        </button>
                    </div>
                    
                    
                </div>
            </form>
            
            <div class="loading" id="loadingDiv">
                <div class="spinner"></div>
                <p>Processando chunks semânticos...</p>
                <small>Aguarde enquanto analisamos e dividimos seu texto.</small>
            </div>
            
            <?php if (!empty($pageData['processingLog']) && $interfaceConfig['show_processing_log']): ?>
                <div class="log">
                    <h4>Log de Processamento:</h4>
                    <?php foreach ($pageData['processingLog'] as $logItem): ?>
                        <div class="log-item"><?= htmlspecialchars($logItem) ?></div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
            
            <?php if (!empty($pageData['result'])): ?>
                <?php 
                $result = $pageData['result']; 
                $quality = getDataSafely($result, 'quality', []);
                $statistics = getDataSafely($result, 'statistics', []);
                $chunks = getDataSafely($result, 'chunks', []);
                $qualityGrade = getDataSafely($quality, 'grade', 'DESCONHECIDA');
                $qualityClass = getQualityCssClass($quality);
                ?>
                
                <div class="success">
                    <strong>Processamento Concluído!</strong> 
                    Texto dividido em <?= count($chunks) ?> chunks com qualidade 
                    <span class="quality-indicator quality-<?= $qualityClass ?>">
                        <?= htmlspecialchars($qualityGrade) ?>
                    </span>
                </div>
                
                <?php if ($interfaceConfig['show_statistics'] && !empty($statistics)): ?>
                    <div class="stats">
                        <div class="stat-item">
                            <div class="stat-number"><?= count($chunks) ?></div>
                            <div class="stat-label">Chunks</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-number"><?= number_format(getDataSafely($statistics, 'chars', 0)) ?></div>
                            <div class="stat-label">Caracteres</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-number"><?= number_format(getDataSafely($statistics, 'estimated_tokens', 0)) ?></div>
                            <div class="stat-label">Tokens</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-number"><?= number_format(getDataSafely($statistics, 'words', 0)) ?></div>
                            <div class="stat-label">Palavras</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-number"><?= round(array_sum(array_column($quality, 'quality_score')) / count($chunks) * 100, 1) ?>%</div>
                            <div class="stat-label">Qualidade</div>
                        </div>
                    </div>
                <?php endif; ?>
                
                <div class="chunks-result">
                    <div class="result-header">
                        <h3>Chunks Gerados:</h3>
                        <div class="export-controls">
                            <button type="button" class="btn btn-export" onclick="exportResult('json')">JSON</button>
                            <button type="button" class="btn btn-export" onclick="exportResult('csv')">CSV</button>
                            <button type="button" class="btn btn-export" onclick="exportResult('txt')">TXT</button>
                        </div>
                    </div>
                    
                    <div class="chunks-output">
                        <button class="copy-button" onclick="copyChunks()" id="copyBtn">Copiar</button>
                        <textarea 
                            id="chunksOutput" 
                            class="chunks-textarea" 
                            readonly
                        ><?= htmlspecialchars(generateSimpleChunksText($chunks)) ?></textarea>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>
    
    <script>
        // Configurações
        const config = {
            minLength: <?= $validationRules['min_text_length'] ?>,
            maxLength: <?= $validationRules['max_text_length'] ?>,
            charsPerToken: <?= $chunkConfig['chars_per_token'] ?>,
            targetChunkSize: <?= $chunkConfig['target_chunk_size'] ?>
        };
        
        // Elementos DOM
        const inputText = document.getElementById('input_text');
        const charCount = document.getElementById('charCount');
        const tokenEstimate = document.getElementById('tokenEstimate');
        const chunkEstimate = document.getElementById('chunkEstimate');
        const submitBtn = document.getElementById('submitBtn');
        const loadingDiv = document.getElementById('loadingDiv');
        const chunkingForm = document.getElementById('chunkingForm');
        
        // Atualizar contadores
        function updateCounts() {
            const count = inputText.value.length;
            const tokens = Math.ceil(count / config.charsPerToken);
            const chunks = Math.ceil(tokens / config.targetChunkSize);
            
            charCount.textContent = count.toLocaleString();
            tokenEstimate.textContent = tokens.toLocaleString();
            chunkEstimate.textContent = chunks.toLocaleString();
            
            // Validação visual
            charCount.classList.remove('warning', 'error');
            
            if (count > config.maxLength) {
                charCount.classList.add('error');
                submitBtn.disabled = true;
            } else if (count < config.minLength && count > 0) {
                charCount.classList.add('warning');
                submitBtn.disabled = false;
            } else {
                submitBtn.disabled = count === 0;
            }
        }
        
        inputText.addEventListener('input', updateCounts);
        updateCounts();
        
        // Limpar formulário
        function clearForm() {
            if (confirm('Limpar todo o texto?')) {
                inputText.value = '';
                updateCounts();
            }
        }
        
        // Copiar chunks
        function copyChunks() {
            const output = document.getElementById('chunksOutput');
            const btn = document.getElementById('copyBtn');
            
            output.select();
            document.execCommand('copy');
            
            btn.textContent = 'Copiado!';
            btn.classList.add('copied');
            
            setTimeout(() => {
                btn.textContent = 'Copiar';
                btn.classList.remove('copied');
            }, 2000);
        }
        
        // Export functions
        function exportResult(format) {
            const form = document.createElement('form');
            form.method = 'POST';
            form.style.display = 'none';
            
            const exportInput = document.createElement('input');
            exportInput.name = 'export_format';
            exportInput.value = format;
            
            const textInput = document.createElement('input');
            textInput.name = 'input_text';
            textInput.value = inputText.value;
            
            const actionInput = document.createElement('input');
            actionInput.name = 'action';
            actionInput.value = 'export';
            
            form.appendChild(exportInput);
            form.appendChild(textInput);
            form.appendChild(actionInput);
            document.body.appendChild(form);
            form.submit();
            document.body.removeChild(form);
        }
        
        // Loading durante submit
        chunkingForm.addEventListener('submit', function() {
            submitBtn.disabled = true;
            submitBtn.textContent = 'Processando...';
            loadingDiv.style.display = 'block';
        });
        
        // Auto-resize textarea
        inputText.addEventListener('input', function() {
            this.style.height = 'auto';
            this.style.height = Math.max(250, this.scrollHeight) + 'px';
        });
        
        // Inicialização
        document.addEventListener('DOMContentLoaded', function() {
            updateCounts();
            
            // Auto-resize da área de output se existir
            const chunksOutput = document.getElementById('chunksOutput');
            if (chunksOutput) {
                chunksOutput.style.height = 'auto';
                chunksOutput.style.height = Math.max(400, chunksOutput.scrollHeight) + 'px';
                
                // Scroll até o resultado
                setTimeout(() => {
                    chunksOutput.scrollIntoView({ behavior: 'smooth', block: 'center' });
                }, 500);
            }
        });
    </script>
</body>
</html>