<?php
/**
 * Processador Principal - Orquestração da Aplicação
 * Arquivo: processor.php
 */

require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../prompt/prompt-normalize.php';
require_once __DIR__ . '/../prompt/prompt-chunking.php';

/**
 * Classe principal para processar texto de transcrições
 */
class TextProcessor {
    
    private $log = [];
    
    /**
     * Processa texto completo: normalização + chunking
     * 
     * @param string $rawText Texto bruto da transcrição
     * @return array Resultado do processamento com chunks e logs
     */
    public function processText($rawText) {
        try {
            $this->addLog("Iniciando processamento do texto...");
            $this->addLog("Tamanho do texto original: " . strlen($rawText) . " caracteres");
            
            // Validação inicial
            if (empty($rawText)) {
                throw new Exception("Texto não pode estar vazio");
            }
            
            if (strlen($rawText) < 100) {
                throw new Exception("Texto muito curto para processamento");
            }
            
            if (strlen($rawText) > 50000) {
                throw new Exception("Texto muito longo (máximo 50.000 caracteres)");
            }
            
            // Etapa 1: Normalização
            $this->addLog("Etapa 1: Iniciando normalização do texto...");
            $normalizedText = $this->normalizeTextStep($rawText);
            $this->addLog("Normalização concluída. Tamanho: " . strlen($normalizedText) . " caracteres");
            
            // Etapa 2: Chunking semântico
            $this->addLog("Etapa 2: Iniciando divisão em chunks semânticos...");
            $chunks = $this->createChunksStep($normalizedText);
            $this->addLog("Chunking concluído. Total de chunks: " . count($chunks));
            
            // Estatísticas finais
            $this->generateStats($chunks);
            
            return [
                'success' => true,
                'normalized_text' => $normalizedText,
                'chunks' => $chunks,
                'stats' => $this->getStats($chunks),
                'log' => $this->log
            ];
            
        } catch (Exception $e) {
            $this->addLog("ERRO: " . $e->getMessage());
            
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'log' => $this->log
            ];
        }
    }
    
    /**
     * Etapa 1: Normalização do texto
     */
    private function normalizeTextStep($rawText) {
        $startTime = microtime(true);
        
        try {
            $normalizedText = normalizeText($rawText);
            
            $duration = round((microtime(true) - $startTime), 2);
            $this->addLog("Tempo de normalização: {$duration}s");
            
            if (empty($normalizedText)) {
                throw new Exception("Texto normalizado está vazio");
            }
            
            return $normalizedText;
            
        } catch (Exception $e) {
            throw new Exception("Falha na normalização: " . $e->getMessage());
        }
    }
    
    /**
     * Etapa 2: Criação de chunks semânticos
     */
    private function createChunksStep($normalizedText) {
        $startTime = microtime(true);
        
        try {
            $chunks = createSemanticChunks($normalizedText);
            $validatedChunks = validateChunks($chunks);
            
            $duration = round((microtime(true) - $startTime), 2);
            $this->addLog("Tempo de chunking: {$duration}s");
            
            if (empty($validatedChunks)) {
                throw new Exception("Nenhum chunk válido foi criado");
            }
            
            return $validatedChunks;
            
        } catch (Exception $e) {
            throw new Exception("Falha na criação de chunks: " . $e->getMessage());
        }
    }
    
    /**
     * Gera estatísticas dos chunks
     */
    private function generateStats($chunks) {
        $totalWords = 0;
        $topics = [];
        $allKeywords = [];
        
        foreach ($chunks as $chunk) {
            // Calcula palavras dinamicamente já que removemos word_count
            $wordCount = str_word_count($chunk['content'] ?? '');
            $totalWords += $wordCount;
            
            if (!empty($chunk['topic'])) {
                $topics[] = $chunk['topic'];
            }
            
            if (is_array($chunk['keywords'])) {
                $allKeywords = array_merge($allKeywords, $chunk['keywords']);
            }
        }
        
        $avgWordsPerChunk = count($chunks) > 0 ? round($totalWords / count($chunks)) : 0;
        $uniqueTopics = array_unique($topics);
        $topKeywords = array_count_values($allKeywords);
        arsort($topKeywords);
        $topKeywords = array_slice($topKeywords, 0, 10, true);
        
        $this->addLog("Estatísticas finais:");
        $this->addLog("- Total de chunks: " . count($chunks));
        $this->addLog("- Total de palavras: $totalWords");
        $this->addLog("- Média de palavras por chunk: $avgWordsPerChunk");
        $this->addLog("- Tópicos únicos: " . count($uniqueTopics));
        $this->addLog("- Principais palavras-chave: " . implode(', ', array_keys($topKeywords)));
    }
    
    /**
     * Retorna estatísticas detalhadas
     */
    private function getStats($chunks) {
        $totalWords = 0;
        $topics = [];
        $allKeywords = [];
        
        foreach ($chunks as $chunk) {
            // Calcula palavras dinamicamente
            $wordCount = str_word_count($chunk['content'] ?? '');
            $totalWords += $wordCount;
            
            if (!empty($chunk['topic'])) {
                $topics[] = $chunk['topic'];
            }
            
            if (is_array($chunk['keywords'])) {
                $allKeywords = array_merge($allKeywords, $chunk['keywords']);
            }
        }
        
        $avgWordsPerChunk = count($chunks) > 0 ? round($totalWords / count($chunks)) : 0;
        $topKeywords = array_count_values($allKeywords);
        arsort($topKeywords);
        $topKeywords = array_slice($topKeywords, 0, 10, true);
        
        return [
            'total_chunks' => count($chunks),
            'total_words' => $totalWords,
            'avg_words_per_chunk' => $avgWordsPerChunk,
            'unique_topics' => array_unique($topics),
            'top_keywords' => $topKeywords
        ];
    }
    
    /**
     * Adiciona entrada ao log
     */
    private function addLog($message) {
        $timestamp = date('H:i:s');
        $this->log[] = "[$timestamp] $message";
    }
    
    /**
     * Exporta chunks para formato ChromaDB
     */
    public function exportForChromaDB($chunks) {
        $export = [
            'documents' => [],
            'metadatas' => [],
            'ids' => []
        ];
        
        foreach ($chunks as $chunk) {
            $export['documents'][] = $chunk['content'];
            $export['metadatas'][] = [
                'topic' => $chunk['topic'],
                'keywords' => $chunk['keywords'] // Mantém como array para melhor usabilidade
            ];
            $export['ids'][] = 'chunk_' . $chunk['id'];
        }
        
        return $export;
    }
}

// Endpoint para processar via POST (apenas quando chamado diretamente)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && basename($_SERVER['SCRIPT_NAME']) === 'processor.php') {
    header('Content-Type: application/json; charset=utf-8');
    
    try {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!isset($input['text']) || empty($input['text'])) {
            throw new Exception("Campo 'text' é obrigatório");
        }
        
        $processor = new TextProcessor();
        $result = $processor->processText($input['text']);
        
        // Se solicitado, adiciona export para ChromaDB
        if (isset($input['export_chromadb']) && $input['export_chromadb'] && $result['success']) {
            $result['chromadb_export'] = $processor->exportForChromaDB($result['chunks']);
        }
        
        echo json_encode($result, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        exit; // Para não continuar executando quando chamado via API
        
    } catch (Exception $e) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'error' => $e->getMessage()
        ], JSON_UNESCAPED_UNICODE);
        exit;
    }
}