<?php
/**
 * Prompt para Divisão em Chunks Semânticos
 * Arquivo: prompt-chunking.php
 */

require_once __DIR__ . '/../config/config.php';

/**
 * Divide texto normalizado em chunks semânticos para embedding
 * 
 * @param string $normalizedText Texto já normalizado
 * @return array Array de chunks semânticos
 */
function createSemanticChunks($normalizedText) {
    $system_message = "Você é um especialista em processamento de linguagem natural e embedding de textos. Sua tarefa é dividir textos normalizados em chunks semânticos otimizados para busca por similaridade em bases de dados vetoriais como ChromaDB.";
    
    $target_size = CHUNK_SIZE_TARGET;
    
    $prompt = "
TEXTO NORMALIZADO PARA DIVIDIR:
==============================
$normalizedText

INSTRUÇÕES PARA CHUNKING SEMÂNTICO:
===================================

1. CRITÉRIOS DE DIVISÃO:
   - Divida por tópicos/assuntos coerentes
   - Mantenha o contexto semântico completo em cada chunk
   - Cada chunk deve ser auto-suficiente para entendimento
   - Tamanho alvo: aproximadamente $target_size palavras por chunk

2. PRESERVAÇÃO DE CONTEXTO:
   - Mantenha falas da completas (não corte no meio)
   - Preserve diálogos relacionados no mesmo chunk
   - Inclua contexto suficiente para compreensão isolada

3. IDENTIFICAÇÃO DE BOUNDARIES NATURAIS:
   - Mudanças de tópico
   - Transições entre diferentes assuntos
   - Pausas naturais na conversa
   - Início/fim de blocos de perguntas e respostas

4. METADADOS POR CHUNK:
   - Identifique o tópico principal
   - Marque se contém fala da Cora
   - Adicione palavras-chave relevantes

5. FORMATO DE SAÍDA:
   Retorne um JSON válido com o seguinte formato:
   {
     \"chunks\": [
       {
         \"id\": 1,
         \"content\": \"texto do chunk\",
         \"topic\": \"tópico principal\",
         \"keywords\": [\"palavra1\", \"palavra2\", \"palavra3\"]
       }
     ]
   }

IMPORTANTE:
- Retorne APENAS o JSON, sem texto adicional
- Garanta que o JSON seja válido
- Não perca nenhum conteúdo do texto original
- Chunks devem cobrir todo o texto com ligeira sobreposição natural nas transições

RESULTADO JSON:
";

    try {
        $response = callOpenAI($prompt, $system_message);
        
        // Limpa possível texto extra e extrai apenas o JSON
        $response = trim($response);
        
        // Procura pelo início do JSON
        $start = strpos($response, '{');
        if ($start !== false) {
            $response = substr($response, $start);
        }
        
        // Procura pelo final do JSON
        $end = strrpos($response, '}');
        if ($end !== false) {
            $response = substr($response, 0, $end + 1);
        }
        
        $chunks_data = json_decode($response, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception("Erro ao decodificar JSON: " . json_last_error_msg());
        }
        
        if (!isset($chunks_data['chunks']) || !is_array($chunks_data['chunks'])) {
            throw new Exception("Formato de resposta inválido: chunks não encontrados");
        }
        
        return $chunks_data['chunks'];
        
    } catch (Exception $e) {
        throw new Exception("Erro na criação de chunks: " . $e->getMessage());
    }
}

/**
 * Valida e corrige chunks se necessário
 * 
 * @param array $chunks Array de chunks
 * @return array Chunks validados
 */
function validateChunks($chunks) {
    $validated = [];
    
    foreach ($chunks as $index => $chunk) {
        // Garante que todos os campos obrigatórios existem
        $validated_chunk = [
            'id' => isset($chunk['id']) ? $chunk['id'] : $index + 1,
            'content' => isset($chunk['content']) ? trim($chunk['content']) : '',
            'topic' => isset($chunk['topic']) ? $chunk['topic'] : 'Sem tópico',
            'keywords' => isset($chunk['keywords']) && is_array($chunk['keywords']) ? $chunk['keywords'] : []
        ];
        
        // Só adiciona se o chunk tem conteúdo
        if (!empty($validated_chunk['content'])) {
            $validated[] = $validated_chunk;
        }
    }
    
    return $validated;
}
?>
